/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
usr_uart_env_t env;
md_uart_init_t init;
uint8_t tx_buf[64];
uint8_t rx_buf[64];

/**
  * @brief  Initializate pin
  * @retval None
  */
void md_uart_pin_init(void)
{
    md_gpio_init_t x;
    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &x);
}

/**
  * @brief  Send data by interrupt
  * @param  buf: Pointer to data.
  * @param  size: Length of data.
  * @retval None
  */
void usr_uart_tx_by_interrupt(uint8_t *buf, uint32_t size)
{
    if ((size == 0) || (env.state & USR_STATE_TX))
        return;

    env.tx_buf  = buf;
    env.tx_cnt  = 0;
    env.tx_size = size;

    SET_BIT(env.state, USR_STATE_TX);
    md_uart_enable_it_tfempty(UARTx);
    return;
}


/**
  * @brief  Receive data by interrupt
  * @param  buf: Pointer to data.
  * @param  size: Length of data.
  * @retval None
  */
void usr_uart_rx_by_interrupt(uint8_t *buf, uint32_t size)
{
    if ((size == 0) || (env.state & USR_STATE_RX))
        return;

    env.rx_buf  = buf;
    env.rx_cnt  = 0;
    env.rx_size = size;

    SET_BIT(env.state, USR_STATE_RX);
    md_uart_set_rx_threshold(UARTx, 0x0);
    md_uart_enable_it_rfth(UARTx);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    int i;
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* Initialize 1ms tick */
    md_init_1ms_tick();
    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);

    /* clear usr_uart_env_t structure */
    memset(&env, 0x0, sizeof(usr_uart_env_t));
    /* clear md_uart_init_t structure */
    memset(&init, 0x0, sizeof(md_uart_init_t));
    /* Initialize pin */
    md_uart_pin_init();
    /* Initialize buffer */
    memset(rx_buf, 0x0, 64);

    for (i = 0; i < 64; ++i)
        tx_buf[i] = i + 1;

    /* Initialize UART */
    init.baud        = 115200;
    init.word_length = MD_UART_WORD_LENGTH_8B;
    init.stop_bits   = MD_UART_STOP_BITS_1;
    init.parity      = MD_UART_PARITY_NONE;
    init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    md_uart_init(UARTx, &init);

    /* Enable interrupt */
    md_mcu_irq_config(UART0_IRQn, 3, ENABLE);

    /* Send data by interrupt */
    usr_uart_tx_by_interrupt(tx_buf, 64);
    /* Receive data by interrupt */
    usr_uart_rx_by_interrupt(rx_buf, 20);

    while (1)
    {
        switch (env.event)
        {
            case USR_EVENT_TX_CPLT:
                /* TX completed */
                break;

            case USR_EVENT_RX_CPLT:
                /* RX completed */
                break;

            case USR_EVENT_TXRX_CPLT:
                /* TX and RX completed */
                break;

            case USR_EVENT_IDLE:
                break;

            default:
                break;
        }

        md_delay_1ms(10000);
        env.event = 0;

        usr_uart_tx_by_interrupt(tx_buf, 64);
        usr_uart_rx_by_interrupt(rx_buf, 20);
    }
}
/**
  * @}
  */
/**
  * @}
  */
