/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

md_uart_init_t init;
uint8_t tx_buf[32];
uint8_t rx_buf[32];

/**
  * @brief  Initializate pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_6;
    x.nodrv = MD_GPIO_OUT_DRIVE_6;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_6;
    x.nodrv = MD_GPIO_OUT_DRIVE_6;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &x);

    return;
}

/**
  * @brief  Send data
  * @param  buf: Pointer to the data
  * @param  size: Length of the data
  * @retval None
  */
void usr_uart_tx(uint8_t *buf, uint32_t size)
{
    while (size--)
    {
        while (!(READ_BIT(UART0->STAT, UART_STAT_TFEMPTY_MSK)));

        UART0->TXBUF = *buf++;
    }

    return;
}

/**
  * @brief  Receive data
  * @param  buf: Pointer to the data
  * @param  size: Length of the data
  * @retval None
  */
void usr_uart_rx(uint8_t *buf, uint32_t size)
{
    while (size--)
    {
        while (READ_BIT(UART0->STAT, UART_STAT_RFEMPTY_MSK));

        *buf++ = UART0->RXBUF;
    }

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* Initialize 1ms tick */
    md_init_1ms_tick();
    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);

    /* clear md_uart_init_t structure */
    memset(&init, 0x0, sizeof(md_uart_init_t));
    /* Initialize pin */
    uart_pin_init();
    /* Initialize buf */
    memset(tx_buf, 0x55, 32);
    memset(tx_buf, 0x0, 32);

    /* Initialize UART */
    init.baud        = 115200;
    init.word_length = MD_UART_WORD_LENGTH_8B;
    init.stop_bits   = MD_UART_STOP_BITS_1;
    init.parity      = MD_UART_PARITY_NONE;
    init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    md_uart_init(UART0, &init);

    /* Send data */
    usr_uart_tx(tx_buf, 32);
    /* Receive data */
    usr_uart_rx(rx_buf, 20);

    while (1)
    {
        md_delay_1ms(1000);
        usr_uart_tx(tx_buf, 8);
    }
}

/**
  * @}
  */
/**
  * @}
  */
