/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  gpio init
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t x;

    /* Initialize sck pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_6;
    x.nodrv = MD_GPIO_OUT_DRIVE_6;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_5;
    md_gpio_init(CHANNEL1_PORT, CHANNEL1_PIN, &x);

    return;
}

/**
  * @brief  Initialize TIMER.
  * @param  tim: Pointer to a TIMER_TypeDef structure
  * @retval None
  */
void pwm_config(TIMER_TypeDef *tim)
{
    uint32_t value;
    md_timer_base_init_t init;
    md_timer_oc_init_t oc_init;

    value = (uint32_t)(md_cmu_get_pclk1_clock() / 2000000) - 1;
    init.prescaler = value;
    init.mode      = MD_TIMER_CNT_MODE_UP;
    init.period    = 100 - 1;
    init.clk_div   = MD_TIMER_CLOCK_DIV1;
    init.re_cnt    = 0;
    md_timer_base_set_config(tim, &init);
    CLEAR_BIT(tim->SMCON, TIMER_SMCON_SMODS_MSK);

    oc_init.ocstate      = MD_TIMER_OCSTATE_ENABLE;
    oc_init.ocnstate     = MD_TIMER_OCSTATE_DISABLE;
    oc_init.oc_mode      = MD_TIMER_OC_MODE_PWM1;
    oc_init.pulse        = init.period >> 2;
    oc_init.oc_polarity  = MD_TIMER_OC_POLARITY_HIGH;
    oc_init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;
    oc_init.oc_fast_en   = DISABLE;
    oc_init.oc_idle      = MD_TIMER_OC_IDLE_RESET;
    oc_init.ocn_idle     = MD_TIMER_OCN_IDLE_RESET;
    md_timer_oc_init(tim, MD_TIMER_CHANNEL_1, &oc_init);

    /* main output enable */
    md_timer_set_main_output_enable(tim, 0x01);
    /* enable timer */
    md_timer_enable_counter_cnten(tim);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Configure system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    /* Initialize 1ms tick */
    md_init_1ms_tick();
    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);

    /* Initialize pin */
    gpio_init();
    /* Initialize timer */
    pwm_config(GP16C2T0);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

