/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint8_t tx_buf[32];
uint8_t rx_buf[32];

/**
  * @brief  Initializate pin of lpuart module.
  * @retval None
  */
void lpuart_pin_init(void)
{
    md_gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_6;
    x.nodrv = MD_GPIO_OUT_DRIVE_6;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_TX_PORT, LPUART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_6;
    x.nodrv = MD_GPIO_OUT_DRIVE_6;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_TTL;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(LPUART0_RX_PORT, LPUART0_RX_PIN, &x);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    int32_t i;
    md_lpuart_init_t init;

    /* Initialize 1ms tick */
    md_init_1ms_tick();
    /* Enable all peripherals clock */
    md_cmu_perh_clock_config(MD_CMU_PERH_ALL, ENABLE);

    /* clear md_lpuart_init_t structure */
    memset(&init, 0x0, sizeof(md_lpuart_init_t));
    /* Initialize pin */
    lpuart_pin_init();

    /* Initialize rx_buf */
    memset(rx_buf, 0x0, 32);

    /* Initialize tx_buf */
    for (i = 0; i < 32; ++i)
        tx_buf[i] = i + 1;

    /* Set LP16T0 clock source(LOSC) */
    SYSCFG_UNLOCK();
    md_cmu_set_lpuart0_clock_source(0x6);
    SYSCFG_LOCK();

    /* Initialize LPUART */
    init.baud        = 2400;
    init.word_length = MD_LPUART_WORD_LENGTH_8B;
    init.stop_bits   = MD_LPUART_STOP_BITS_1;
    init.parity      = MD_LPUART_PARITY_NONE;
    init.mode        = MD_LPUART_MODE_UART;
    init.fctl        = MD_LPUART_FLOW_CTL_DISABLE;
    md_lpuart_init(LPUART0, &init);

    /* Send a message */
    for (i = 0; i < 32; ++i)
    {
        while (!(READ_BIT(LPUART0->STAT, LPUART_STAT_TXEMP_MSK)));

        md_lpuart_send_data(LPUART0, tx_buf[i]);

        while ((READ_BIT(LPUART0->STAT, LPUART_STAT_TXEMP_MSK)));
    }

    /* Receive a message */
    for (i = 0; i < 8; ++i)
    {
        while ((READ_BIT(LPUART0->STAT, LPUART_STAT_RXEMP_MSK)));

        rx_buf[i] = (uint8_t)md_lpuart_recv_data(LPUART0);
    }

    while (1)
    {
        md_delay_1ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */
