/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main File for Demo
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_UART
  * @{
  */

uart_handle_t h_uart;
uint8_t tx_buf[32] __attribute__((at(0x20000100)));
uint8_t rx_buf[32] __attribute__((at(0x20000200)));
uint8_t uart0_tx_flag = 0;
uint8_t uart0_rx_flag = 0;

extern dma_descriptor_t dma0_ctrl_base[28];

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(UART0_TX_PORT, UART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(UART0_RX_PORT, UART0_RX_PIN, &x);

    return;
}

void dma_restart__(dma_handle_t *hperh, void *src, void *dst, uint16_t size)
{
    src = dst;
    src = src;
    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_error(uart_handle_t *arg)
{
    return;
}

void usr_dma_init(void)
{
    h_uart.hdmatx.perh     = DMA0;
    h_uart.hdmatx.cplt_cbk = NULL;
    h_uart.hdmatx.cplt_arg = NULL;
    h_uart.hdmatx.err_cbk  = NULL;
    h_uart.hdmatx.err_arg  = NULL;

    ald_dma_config_struct(&h_uart.hdmatx.config);
    h_uart.hdmatx.config.src     = (void *)0x20000100;
    h_uart.hdmatx.config.dst     = (void *)&h_uart.perh->TXBUF;
    h_uart.hdmatx.config.size    = 0;
    h_uart.hdmatx.config.src_inc = DMA_DATA_INC_BYTE;
    h_uart.hdmatx.config.dst_inc = DMA_DATA_INC_NONE;
    h_uart.hdmatx.config.msigsel = DMA_MSIGSEL_UART_TXEMPTY;
    h_uart.hdmatx.config.msel    = DMA_MSEL_UART0;
    h_uart.hdmatx.config.burst   = ENABLE;
    h_uart.hdmatx.config.channel = UART0_TX_DMA_CH;
    ald_dma_config_basic(&h_uart.hdmatx);

    DMA0->CHENCLR = (0x1 << UART0_TX_DMA_CH);

    h_uart.hdmarx.perh     = DMA0;
    h_uart.hdmarx.cplt_cbk = NULL;
    h_uart.hdmarx.cplt_arg = NULL;
    h_uart.hdmarx.err_cbk  = NULL;
    h_uart.hdmarx.err_arg  = NULL;

    ald_dma_config_struct(&h_uart.hdmarx.config);
    h_uart.hdmarx.config.src     = (void *)&h_uart.perh->RXBUF;
    h_uart.hdmarx.config.dst     = (void *)0x20000200;
    h_uart.hdmarx.config.size    = 0;
    h_uart.hdmarx.config.src_inc = DMA_DATA_INC_NONE;
    h_uart.hdmarx.config.dst_inc = DMA_DATA_INC_BYTE;
    h_uart.hdmarx.config.msigsel = DMA_MSIGSEL_UART_RNR;
    h_uart.hdmarx.config.msel    = DMA_MSEL_UART0;
    h_uart.hdmarx.config.burst   = ENABLE;
    h_uart.hdmarx.config.channel = UART0_RX_DMA_CH;
    ald_dma_config_basic(&h_uart.hdmarx);
    DMA0->CHENCLR = (0x1 << UART0_RX_DMA_CH);

    return;
}

void usr_uart0_tx_by_dma(uint8_t *buf, uint16_t size)
{
    uart0_tx_flag = 0;

    dma0_ctrl_base[UART0_TX_DMA_CH].src = (void *)(buf + size - 1);
    dma0_ctrl_base[UART0_TX_DMA_CH].ctrl.n_minus_1 = size - 1;
    dma0_ctrl_base[UART0_TX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
    WRITE_REG(DMA0->CHENSET, (1 << UART0_TX_DMA_CH));
    SET_BIT(h_uart.perh->MCON, UART_MCON_TXDMAEN_MSK);
    return;
}

void usr_uart0_rx_by_dma(uint8_t *buf, uint16_t size)
{
    uart0_rx_flag = 0;

    dma0_ctrl_base[UART0_RX_DMA_CH].dst = (void *)(buf + size - 1);
    dma0_ctrl_base[UART0_RX_DMA_CH].ctrl.n_minus_1 = size - 1;
    dma0_ctrl_base[UART0_RX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
    WRITE_REG(DMA0->CHENSET, (1 << UART0_RX_DMA_CH));
    SET_BIT(h_uart.perh->MCON, UART_MCON_RXDMAEN_MSK);
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    int i;
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Enable uart interrupt */
//  ald_mcu_irq_config(UART1_IRQn, 3, ENABLE);
    /* clear uart_handle_t structure */
    memset(&h_uart, 0x0, sizeof(h_uart));
    /* Initialize pin */
    uart_pin_init();
    /* Initialize tx_buf */
    memset(tx_buf, 0x55, 32);
    memset(rx_buf, 0x0, 32);

    for (i = 0; i < 32; ++i)
    {
        tx_buf[i] = i + 1;
    }

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = UART_STOP_BITS_1;
    h_uart.init.parity      = UART_PARITY_NONE;
    h_uart.init.mode        = UART_MODE_UART;
    h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = uart_send_complete;
    h_uart.rx_cplt_cbk      = uart_recv_complete;
    h_uart.error_cbk        = uart_error;
    ald_uart_init(&h_uart);

    usr_dma_init();
    usr_uart0_tx_by_dma(tx_buf, 32);
    usr_uart0_rx_by_dma(rx_buf, 20);

    while (1)
    {
        if (uart0_rx_flag == 1)
        {
            usr_uart0_tx_by_dma(tx_buf, 17);
            usr_uart0_rx_by_dma(rx_buf, 20);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */
