/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
timer_handle_t h_tim;
dma_handle_t hperh;
timer_clock_config_t clock;
timer_ic_init_t init;
timer_slave_config_t slave_config;

/* duty/cycle data buffer */
#define BUFFER_SIZE 32
uint16_t duty_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0};
uint16_t cycle_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0};

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode = GPIO_MODE_INPUT;
    x.pupd = GPIO_PUSH_UP;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(TIMER_INPUTCAPTURE_PORT, TIMER_INPUTCAPTURE_PIN, &x);
    ald_gpio_init(GPIOA, GPIO_PIN_2, &x);
}

/**
  * @brief  Initializate DMA chanel0 and channel1.
  * @retval None
  */
void init_dma(void)
{
    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh.config);

    hperh.config.src     = (void *)(&GP16C4T0->CCVAL1);
    hperh.config.dst     = (void *)cycle_buffer;
    hperh.config.size    = BUFFER_SIZE;
    hperh.config.data_width = DMA_DATA_SIZE_HALFWORD;
    hperh.config.src_inc = DMA_DATA_INC_NONE;
    hperh.config.dst_inc = DMA_DATA_INC_HALFWORD;
    hperh.config.R_power = DMA_R_POWER_1;

    hperh.config.msel    = DMA_MSEL_TIMER6;
    hperh.config.msigsel = DMA_MSIGSEL_TIMER_CH1;

    hperh.cplt_cbk       = NULL;
    hperh.cplt_arg       = NULL;
    hperh.err_cbk        = NULL;
    hperh.err_arg        = NULL;
    hperh.config.channel = 0;
    ald_dma_config_basic(&hperh);

    hperh.perh = DMA0;
    ald_dma_config_struct(&hperh.config);

    hperh.config.src     = (void *)(&GP16C4T0->CCVAL2);
    hperh.config.dst     = (void *)duty_buffer;
    hperh.config.size    = BUFFER_SIZE;
    hperh.config.data_width = DMA_DATA_SIZE_HALFWORD;
    hperh.config.src_inc = DMA_DATA_INC_NONE;
    hperh.config.dst_inc = DMA_DATA_INC_HALFWORD;
    hperh.config.R_power = DMA_R_POWER_1;

    hperh.config.msel    = DMA_MSEL_TIMER6;
    hperh.config.msigsel = DMA_MSIGSEL_TIMER_CH2;

    hperh.cplt_cbk       = NULL;
    hperh.cplt_arg       = NULL;
    hperh.err_cbk        = NULL;
    hperh.err_arg        = NULL;
    hperh.config.channel = 1;
    ald_dma_config_basic(&hperh);
}

/**
  * @brief  Main function
  * @retval Status
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    /* Initialize pin and set the default state*/
    timer_pin_init();

    /* Initialize GP16C4T0 */
    h_tim.perh           = GP16C4T0;
    h_tim.init.prescaler = 0;
    h_tim.init.mode      = TIMER_CNT_MODE_UP;
    h_tim.init.period    = 0xFFFF;
    h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = NULL;
    ald_timer_ic_init(&h_tim);

    /* Initialize clock source */
    clock.source     = TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &clock);

    /* Initialize input capture channel */
    init.polarity = TIMER_IC_POLARITY_RISE;
    init.sel      = TIMER_IC_SEL_DIRECT;
    init.psc      = TIMER_IC_PSC_DIV1;
    init.filter   = 3;
    ald_timer_ic_config_channel(&h_tim, &init, TIMER_CHANNEL_1);

    /* Initialize input capture channel */
    init.polarity = TIMER_IC_POLARITY_FALL;
    init.sel      = TIMER_IC_SEL_INDIRECT;
    init.psc      = TIMER_IC_PSC_DIV1;
    init.filter   = 2;
    ald_timer_ic_config_channel(&h_tim, &init, TIMER_CHANNEL_2);

    slave_config.mode   = TIMER_MODE_RESET;
    slave_config.input  = TIMER_TS_TI1FP1;
    slave_config.polarity   = TIMER_CLK_POLARITY_RISE;
    slave_config.psc        = TIMER_ETR_PSC_DIV1;
    slave_config.filter     = 4;
    ald_timer_slave_config_sync(&h_tim, &slave_config);

    init_dma();

    TIMER_CCx_DMA_REQ_CCx(&h_tim);
    ald_timer_dma_req_config(&h_tim, TIMER_DMA_CC1, ENABLE);
    ald_timer_dma_req_config(&h_tim, TIMER_DMA_CC2, ENABLE);

    /* Start input capture */
    ald_timer_ic_start(&h_tim, TIMER_CHANNEL_1);
    ald_timer_ic_start(&h_tim, TIMER_CHANNEL_2);

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */
