/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
timer_handle_t h_tim1;
pis_handle_t h_pis;
adc_handle_t h_adc;
adc_nch_conf_t nor_config;
float nor_value = 0;
uint16_t value[48];
int32_t idx = 0;

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void normal_convert_complete(adc_handle_t *arg)
{
    idx = idx == 48 ? 0 : idx;
    value[idx++] = ald_adc_normal_get_value(&h_adc);

    return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void error_complete(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
void ovr_complete(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Initializate ADC pin.
  * @retval None
  */

void adc_pin_init(void)
{
    gpio_init_t x;

    /* Initialize GPIO_PIN_0 as input */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.flt  = GPIO_FILTER_DISABLE;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_0;
    ald_gpio_init(GPIOA, GPIO_PIN_0, &x);
}

/**
  * @brief  Initializate ADC module.
  * @retval None
  */

void pis_adc_init()
{
    /* clear lpuart_handle_t structure */
    memset(&h_adc, 0x0, sizeof(adc_handle_t));
    /* clear adc_channel_conf_t structure */
    memset(&nor_config, 0x0, sizeof(nor_config));
    /* Enable adc interrupt */
    ald_mcu_irq_config(ADC_ACMP_IRQn, 3, ENABLE);

    /* Initialize adc */
    h_adc.perh            = ADC0;
    h_adc.init.align      = ADC_DATAALIGN_RIGHT;
    h_adc.init.scan       = DISABLE;
    h_adc.init.cont       = DISABLE;
    h_adc.init.nch_nr     = ADC_NCH_NR_1;
    h_adc.init.disc       = ADC_NCH_DISC_EN;
    h_adc.init.disc_nr    = ADC_DISC_NR_1;
    h_adc.init.data_bit   = ADC_CONV_BIT_12;
    h_adc.init.div        = ADC_CKDIV_128;
    h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
    h_adc.init.n_ref      = ADC_NEG_REF_VSS;
    h_adc.init.p_ref      = ADC_POS_REF_VDD;
    h_adc.normal_cplt_cbk = normal_convert_complete;
    h_adc.insert_cplt_cbk = NULL;
    h_adc.wdg_cbk         = NULL;
    h_adc.error_cbk       = error_complete;
    h_adc.ovr_cbk         = ovr_complete;
    ald_adc_init(&h_adc);

    /* adc normal external trigger rise edge*/
    ADC_NETS_ENABLE(&h_adc, ADC_ETS_RISE);

    /* Initialize regular convert channel */
    nor_config.ch   = ADC_CHANNEL_0;
    nor_config.idx  = ADC_NCH_IDX_1;
    nor_config.samp = ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &nor_config);

    /* Initialize PIS */
    memset(&h_pis, 0, sizeof(pis_handle_t));

    h_pis.perh               = PIS;
    h_pis.init.producer_src  = PIS_BS16T0_UPDATA;
    h_pis.init.producer_clk  = PIS_CLK_PCLK1;
    h_pis.init.producer_edge = PIS_EDGE_NONE;
    h_pis.init.consumer_trig = PIS_CH6_ADC0_NORMAL;
    h_pis.init.consumer_clk  = PIS_CLK_PCLK2;

    ald_pis_create(&h_pis);
}

/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
    /* clear timer_handle_t structure */
    memset(&h_tim1, 0, sizeof(timer_handle_t));

    /* Initialize BS16T0 */
    h_tim1.perh           = BS16T0;
    h_tim1.init.prescaler = 1000 - 1;
    h_tim1.init.mode      = TIMER_CNT_MODE_UP;
    h_tim1.init.period    = 2000;
    h_tim1.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim1.init.re_cnt    = 0;
    ald_timer_base_init(&h_tim1);
}

/**
  * @brief  Start timer0 and timer2 module.
  * @retval None
  */
void pis_tim_start(void)
{
    ald_timer_base_start(&h_tim1);
}

/**
  * @brief  Stop timer1 module.
  * @retval None
  */
void pis_tim_stop(void)
{
    ald_timer_base_stop(&h_tim1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    /* Enable all peripherals */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Initialize timer */
    pis_tim_init();
    /* Initialize adc/pis */
    adc_pin_init();
    pis_adc_init();
    /* Enable adc interrupt */
    ald_adc_clear_flag_status(&h_adc, ADC_FLAG_NCH);
    ald_adc_interrupt_config(&h_adc, ADC_IT_NCH, ENABLE);
    /* start pis trigger adc normal sample */
    pis_tim_start();

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */
