/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

pis_handle_t h_pis;
timer_handle_t h_tim1;
timer_handle_t h_tim2;
timer_clock_config_t tim_clock;
uint32_t cnt = 0;

/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
    /* clear timer_handle_t structure */
    memset(&h_tim1, 0, sizeof(timer_handle_t));
    /* clear timer_handle_t structure */
    memset(&h_tim2, 0, sizeof(timer_handle_t));
    /* clear timer_clock_config_t structure */
    memset(&tim_clock, 0, sizeof(timer_clock_config_t));

    /* Initialize BS16T0 */
    h_tim1.perh           = BS16T0;
    h_tim1.init.prescaler = 512;
    h_tim1.init.mode      = TIMER_CNT_MODE_UP;
    h_tim1.init.period    = 100;
    h_tim1.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim1.init.re_cnt    = 0;
    ald_timer_base_init(&h_tim1);

    /* Initialize GP16C2T0 */
    h_tim2.perh           = GP16C2T0;
    h_tim2.init.prescaler = 0x01;
    h_tim2.init.mode      = TIMER_CNT_MODE_UP;
    h_tim2.init.period    = 200;
    h_tim2.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim2.init.re_cnt    = 0;
    ald_timer_base_init(&h_tim2);

    /* Initialize clock source */
    tim_clock.source   = TIMER_SRC_ITR0;
    tim_clock.psc      = TIMER_ETR_PSC_DIV1;
    tim_clock.filter   = 0;
    tim_clock.polarity = TIMER_CLK_POLARITY_RISE;
    ald_timer_config_clock_source(&h_tim2, &tim_clock);

    /* Enable GP16C2T0 "UPDATE" interrupt */
    ald_mcu_irq_config(GP16C2T0_IRQn, 2, ENABLE);
    ald_timer_interrupt_config(&h_tim2, TIMER_IT_UPDATE, ENABLE);
    return;
}

/**
  * @brief  Start BS16T0 and GP16C2T0 module.
  * @retval None
  */
void pis_tim_start(void)
{
    ald_timer_base_start(&h_tim1);
    ald_timer_base_start(&h_tim2);
}

/**
  * @brief  Stop BS16T0 module.
  * @retval None
  */
void pis_tim_stop(void)
{
    ald_timer_base_stop(&h_tim1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    /* Enable all peripherals */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Clear pis_handle_t struct */
    memset(&h_pis, 0, sizeof(pis_handle_t));
    /* Initialize timer */
    pis_tim_init();

    /* Initialize PIS */
    h_pis.perh               = PIS;
    h_pis.init.producer_src  = PIS_BS16T0_UPDATA;
    h_pis.init.producer_clk  = PIS_CLK_PCLK1;
    h_pis.init.producer_edge = PIS_EDGE_NONE;
    h_pis.init.consumer_trig = PIS_CH4_GP16C2T0_ITR0;
    h_pis.init.consumer_clk  = PIS_CLK_PCLK1;
    ald_pis_create(&h_pis);

    /* Start PIS */
    pis_tim_start();

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */

/**
  * @}
  */
