/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uart_handle_t h_uart;
dma_config_t config0;
dma_config_t config1;
volatile uint8_t __flag = 0;
volatile uint8_t __nr = 0;

uint8_t src0_buf[32];
uint8_t src1_buf[32];
uint8_t src2_buf[32];
uint8_t src3_buf[32];

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_0_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(UART0_TX_PORT, UART0_TX_PIN, &x);

    /* Initialize rx pin */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_0_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(UART0_RX_PORT, UART0_RX_PIN, &x);

    return;
}

/**
  * @brief  DMA send complete.
  * @param  arg: Pointer.
  * @retval None.
  */
void dma_complete(void *arg)
{
    if (++__nr == 4)
        ald_uart_dma_req_config(&h_uart, UART_DMA_REQ_TX, DISABLE);

    __flag = 1;
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    /* Enable GPIO/UART0 clock */
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_UART0, ENABLE);

    /* Enable uart interrupt */
    ald_mcu_irq_config(UART0_IRQn, 2, ENABLE);
    /* clear uart_handle_t structure */
    memset(&h_uart, 0x0, sizeof(h_uart));
    /* Initialize pin */
    uart_pin_init();
    /* Initialize data */
    memset(src0_buf, 0x22, 32);
    memset(src1_buf, 0x33, 32);
    memset(src2_buf, 0x44, 32);
    memset(src3_buf, 0x55, 32);

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = UART_STOP_BITS_1;
    h_uart.init.parity      = UART_PARITY_NONE;
    h_uart.init.mode        = UART_MODE_UART;
    h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = NULL;
    h_uart.error_cbk        = NULL;
    ald_uart_init(&h_uart);

    /* First descriptor using the primary structure */
    memset(&config0, 0x0, sizeof(dma_config_t));
    config0.src        = src0_buf;
    config0.dst        = (void *)&UART0->TXBUF;
    config0.size       = 32;
    config0.R_power    = DMA_R_POWER_1;
    config0.data_width = DMA_DATA_SIZE_BYTE;
    config0.src_inc    = DMA_DATA_INC_BYTE;;
    config0.dst_inc    = DMA_DATA_INC_NONE;
    config0.primary    = ENABLE;
    config0.burst      = ENABLE;
    config0.high_prio  = DISABLE;
    config0.interrupt  = ENABLE;
    config0.channel    = 0;
    config0.msel       = DMA_MSEL_UART0;
    config0.msigsel    = DMA_MSIGSEL_UART_TXEMPTY;
    ald_dma_config_ping_pong(DMA0, &config0, 1, dma_complete);
    ald_uart_dma_req_config(&h_uart, UART_DMA_REQ_TX, ENABLE);

    /* Second descriptor using the alternate structure */
    memset(&config1, 0x0, sizeof(dma_config_t));
    config1.src        = src1_buf;
    config1.dst        = (void *)&UART0->TXBUF;
    config1.size       = 32;
    config1.R_power    = DMA_R_POWER_1;
    config1.data_width = DMA_DATA_SIZE_BYTE;
    config1.src_inc    = DMA_DATA_INC_BYTE;;
    config1.dst_inc    = DMA_DATA_INC_NONE;
    config1.primary    = DISABLE;
    config1.channel    = 0;
    ald_dma_config_ping_pong(DMA0, &config1, 0, dma_complete);

    /* Wait for the first transfer to complete */
    while (!__flag);

    __flag = 0;

    /* Third descriptor using the primary structure */
    memset(&config0, 0x0, sizeof(dma_config_t));
    config0.src        = src2_buf;
    config0.dst        = (void *)&UART0->TXBUF;
    config0.size       = 32;
    config0.R_power    = DMA_R_POWER_1;
    config0.data_width = DMA_DATA_SIZE_BYTE;
    config0.src_inc    = DMA_DATA_INC_BYTE;;
    config0.dst_inc    = DMA_DATA_INC_NONE;
    config0.primary    = ENABLE;
    config0.channel    = 0;
    ald_dma_config_ping_pong(DMA0, &config0, 0, dma_complete);

    /* Wait for the second transfer to complete */
    while (!__flag);

    __flag = 0;

    /* Fourth descriptor using the alternate structure */
    memset(&config1, 0x0, sizeof(dma_config_t));
    config1.src        = src3_buf;
    config1.dst        = (void *)&UART0->TXBUF;
    config1.size       = 32;
    config1.R_power    = DMA_R_POWER_1;
    config1.data_width = DMA_DATA_SIZE_BYTE;
    config1.src_inc    = DMA_DATA_INC_BYTE;;
    config1.dst_inc    = DMA_DATA_INC_NONE;
    config1.primary    = DISABLE;
    config1.channel    = 0;
    ald_dma_config_ping_pong(DMA0, &config1, 0, dma_complete);

    /* Wait for the third transfer to complete */
    while (!__flag);

    __flag = 0;

    /* Wait for the fourth transfer to complete */
    while (!__flag);

    __flag = 0;

    /* Transfer complete */

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */
