/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0.1
  * @date    28 Apr 2022
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *          28 Apr 2022     AE Team         modify the system clock
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint32_t nm_value;
adc_handle_t h_adc;
uart_handle_t h_uart;
char uart_buf[64];
uint32_t _cnt = 0;

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(uart_buf, 64, fmt, args);
    va_end(args);

    ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
    return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = UART_FUNC_TX;
    ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

    /* Initialize rx pin */
    x.mode  = GPIO_MODE_INPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = UART_FUNC_RX;
    ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

    /* Initialize uart */
    h_uart.perh             = UART0;
    h_uart.init.baud        = 115200;
    h_uart.init.word_length = UART_WORD_LENGTH_8B;
    h_uart.init.stop_bits   = UART_STOP_BITS_1;
    h_uart.init.parity      = UART_PARITY_NONE;
    h_uart.init.mode        = UART_MODE_UART;
    h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    h_uart.tx_cplt_cbk      = NULL;
    h_uart.rx_cplt_cbk      = NULL;
    h_uart.error_cbk        = NULL;
    ald_uart_init(&h_uart);

    return;
}

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void normal_complete_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void insert_complete_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void wdg_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void error_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void ovr_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
    adc_nch_conf_t nm_config;

    /* clear structures */
    memset(&h_adc, 0x0, sizeof(adc_handle_t));
    memset(&nm_config, 0x0, sizeof(adc_nch_conf_t));

    /* Initialize adc */
    h_adc.perh            = ADC0;
    h_adc.init.align      = ADC_DATAALIGN_RIGHT;
    h_adc.init.scan       = DISABLE;
    h_adc.init.cont       = DISABLE;
    h_adc.init.nch_nr     = ADC_NCH_NR_1;
    h_adc.init.disc       = ADC_ALL_DISABLE;
    h_adc.init.disc_nr    = ADC_DISC_NR_1;
    h_adc.init.data_bit   = ADC_CONV_BIT_12;
    h_adc.init.div        = ADC_CKDIV_32;
    h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
    h_adc.init.n_ref      = ADC_NEG_REF_VSS;
    h_adc.init.p_ref      = ADC_POS_REF_2V;
    h_adc.normal_cplt_cbk = normal_complete_cbk;
    h_adc.insert_cplt_cbk = insert_complete_cbk;
    h_adc.wdg_cbk         = wdg_cbk;
    h_adc.error_cbk       = error_cbk;
    h_adc.ovr_cbk         = ovr_cbk;
    ald_adc_init(&h_adc);

    /* Initialize normal convert channel */
    nm_config.ch   = ADC_CHANNEL_16;
    nm_config.idx  = ADC_NCH_IDX_1;
    nm_config.samp = ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &nm_config);

    return;
}

/**
  * @brief  Get the average value of the normal channel.
  * @retval Average voltage.
  */
uint32_t get_normal_average_voltage(void)
{
    uint32_t i, tmp, min = 0xFFFF, max = 0, vol, sum = 0;

    for (i = 0; i < 18; ++i)
    {
        /* Start normal convert */
        ald_adc_normal_start(&h_adc);

        /* Polling convert finish and read result */
        if (ald_adc_normal_poll_for_conversion(&h_adc, 5000) == OK)
            tmp = ald_adc_normal_get_value(&h_adc);
        else
            return 0;

        max  = tmp > max ? tmp : max;
        min  = tmp < min ? tmp : min;
        sum += tmp;
    }

    sum = sum - max - min;
    vol = sum >> 4;

    return vol;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    float Temperature;

    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

    /* Enable peripheral clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Initialize stdio */
    uart_stdio_init();

    /* Initialize ADC */
    adc_init();

    printf_e("System start...\r\n");
    printf_e("Index Normal Tempture\r\n");
    printf_e("--------------------------\r\n");

    nm_value = get_normal_average_voltage();

    Temperature = ald_adc_tsensor_get_temperature(nm_value);

    printf_e("%-5d 0x%-6x %.2f\r\n", ++_cnt, nm_value, Temperature);

    while (1)
    {
        ald_delay_ms(1000);

        nm_value = get_normal_average_voltage();

        Temperature = ald_adc_tsensor_get_temperature(nm_value);

        printf_e("%-5d 0x%-6x %.2f\r\n", ++_cnt, nm_value, Temperature);
    }
}

/**
  * @}
  */

/**
  * @}
  */
