/*
 * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author        Notes
 * 2019-01-23     wangyq        the first version
 */

#include <rthw.h>
#include <rtthread.h>
#include <rtdevice.h>
#include "board.h"
#include "drv_uart.h"
#include <ald_gpio.h>
#include <ald_uart.h>

#ifdef RT_USING_SERIAL

/* es32 uart driver */
struct es32_uart
{
    uart_handle_t huart;
    IRQn_Type irq;
};

static rt_err_t es32h0x_configure(struct rt_serial_device *serial, struct serial_configure *cfg)
{
    gpio_init_t gpio_initstructure;
    struct es32_uart *uart;
    RT_ASSERT(serial != RT_NULL);
    RT_ASSERT(cfg != RT_NULL);
    uart = (struct es32_uart *)serial->parent.user_data;

    /* Initialize tx pin */
    gpio_initstructure.mode = GPIO_MODE_OUTPUT;
    gpio_initstructure.odos = GPIO_PUSH_PULL;
    gpio_initstructure.pupd = GPIO_PUSH_UP;
    gpio_initstructure.podrv = GPIO_OUT_DRIVE_1;
    gpio_initstructure.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_initstructure.flt  = GPIO_FILTER_DISABLE;
    gpio_initstructure.type = GPIO_TYPE_TTL;
    gpio_initstructure.func = GPIO_FUNC_2;
    ald_gpio_init(GPIOB, GPIO_PIN_6, &gpio_initstructure);
    /* Initialize rx pin ,the same as txpin except mode */
    gpio_initstructure.mode = GPIO_MODE_INPUT;
    ald_gpio_init(GPIOB, GPIO_PIN_7, &gpio_initstructure);
	/* uart0 gpio init */

    uart->huart.init.mode        = UART_MODE_UART;
    uart->huart.init.baud        = cfg->baud_rate;
	uart->huart.init.stop_bits	 = (uart_stop_bits_t)cfg->stop_bits;
    uart->huart.init.word_length = (uart_word_length_t)(8 - cfg->data_bits);
    uart->huart.init.parity = (uart_parity_t)(cfg->parity == PARITY_EVEN ? UART_PARITY_EVEN : cfg->parity);
    uart->huart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    ald_uart_init(&uart->huart);

    if (cfg->bit_order == BIT_ORDER_MSB)
    {
        UART_MSB_FIRST_ENABLE(&uart->huart);
    }
    else
    {
        UART_MSB_FIRST_DISABLE(&uart->huart);
    }

    if (cfg->invert == NRZ_INVERTED)
    {
        UART_DATA_INV_ENABLE(&uart->huart);
    }
    else
    {
        UART_DATA_INV_DISABLE(&uart->huart);
    }

    /* enable rx int */
    ald_uart_interrupt_config(&uart->huart, UART_IT_RFTH, ENABLE);

    return RT_EOK;
}

static rt_err_t es32h0x_control(struct rt_serial_device *serial, int cmd, void *arg)
{
    struct es32_uart *uart;
    RT_ASSERT(serial != RT_NULL);

    uart = (struct es32_uart *)serial->parent.user_data;
    switch (cmd)
    {
    case RT_DEVICE_CTRL_CLR_INT:
        /* disable rx irq */
        NVIC_DisableIRQ(uart->irq);
        /* disable interrupt */
        ald_uart_interrupt_config(&uart->huart, UART_IT_RFTH, DISABLE);
        break;

    case RT_DEVICE_CTRL_SET_INT:
        /* enable rx irq */
        NVIC_EnableIRQ(uart->irq);
        /* enable interrupt */
		ald_uart_interrupt_config(&uart->huart, UART_IT_RFTH, ENABLE);
        break;
    }

    return RT_EOK;
}

static int es32h0x_putc(struct rt_serial_device *serial, char c)
{
    struct es32_uart *uart;
    RT_ASSERT(serial != RT_NULL);
    uart = (struct es32_uart *)serial->parent.user_data;

    while (!(uart->huart.perh->STAT & (1U << 15))) ;
    WRITE_REG(uart->huart.perh->TXBUF, c);
    return 1;
}

static int es32h0x_getc(struct rt_serial_device *serial)
{
    int ch = -1;
    struct es32_uart *uart;

    RT_ASSERT(serial != RT_NULL);
    uart = (struct es32_uart *)serial->parent.user_data;

    if (uart->huart.perh->STAT & (1U << 9))
    {
        ch = (uint8_t)(uart->huart.perh->RXBUF & 0xFF);
    }

    return ch;
}

static void uart_isr(struct rt_serial_device *serial)
{
	struct es32_uart *uart = (struct es32_uart *) serial->parent.user_data;
	RT_ASSERT(uart != RT_NULL);
	if((ald_uart_get_mask_flag_status(&uart->huart, UART_IF_RFTH)) != RESET)
	{
        rt_hw_serial_isr(serial, RT_SERIAL_EVENT_RX_IND);	
        ald_uart_clear_flag_status(&uart->huart, UART_IF_RFTH);		
	}
}

static const struct rt_uart_ops es32h0x_uart_ops =
{
    es32h0x_configure,
    es32h0x_control,
    es32h0x_putc,
    es32h0x_getc,
};

/* UART0 device driver structure */
struct es32_uart uart0 =
{
    {UART0},
    UART0_IRQn
};

struct rt_serial_device serial0;

void UART0_Handler(void)
{
    /* enter interrupt */
    rt_interrupt_enter();
	uart_isr(&serial0);
    /* leave interrupt */
    rt_interrupt_leave();
}

int rt_hw_uart_init(void)
{
    struct es32_uart *uart;
    struct serial_configure config = RT_SERIAL_CONFIG_DEFAULT;

    uart = &uart0;
    serial0.ops = &es32h0x_uart_ops;
    serial0.config = config;

    /* register UART0 device */
    rt_hw_serial_register(&serial0, "uart0",
                          RT_DEVICE_FLAG_RDWR | RT_DEVICE_FLAG_INT_RX,
                          uart);

    return 0;
}
INIT_BOARD_EXPORT(rt_hw_uart_init);

#endif
