/**
  *********************************************************************************
  * @file   boot_shell.c
  * @brief  shell module.
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include "boot_shell.h"
#include "boot_flash.h"


/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Shell Shell
  * @brief    Bootloader Shell Module
  * @{
  */
/**
  * @defgroup Shell_Public_Variables Public Variable
  * @brief    Shell Public Variable
  * @{
  */
boot_shell_env_t boot_shell_env;
char uart_buf[64];
/**
  * @}
  */

/**
  * @defgroup Shell_Private_Funtctions Private Functions
  * @brief    Shell Private Functions
  * @{
  */
/**
  * @brief  Refreshs the command cache.
  * @param  cmd: New command
  * @retval None
  */
static void shell_cmd_refresh(char *cmd)
{
    if (!(strcmp(boot_shell_env.cmd, boot_shell_env.cmd_cache0)))
    {
        return;
    }

    memcpy(boot_shell_env.cmd_cache2, boot_shell_env.cmd_cache1, 64);
    memcpy(boot_shell_env.cmd_cache1, boot_shell_env.cmd_cache0, 64);
    memcpy(boot_shell_env.cmd_cache0, cmd, 64);

    return;
}

/**
  * @brief  Handler for input shell command.
  * @param  cmd: input comamd by shell
  * @retval result of execute cmd @ref shell_cmd_exec_t
  */
static shell_cmd_exec_t shell_cmd_exec(char *cmd)
{
    char *c = cmd;
    char i, idx = 0;
    char l_cmd = 0;
    char _arg[16] = {0};
    char _cmd[16] = {0};
    uint32_t arg[4] = {0};
    boot_cmd_ent_t *ent = NULL;
    list_head_t *pos;

    if (*c == '\0')
    {
        printf_e("\r\nes#");
        return CMD_NULL;
    }

    while (*c)
    {
        if (*c == ' ')
        {
            c++;
            continue;
        }

        i = 0;
	memset(_arg, 0x0, 16);

        if (idx == 0)
        {
            while (*c != ' ' && *c && i < 16)
                _cmd[i++] = *c++;
        }
        else
        {
            while (*c != ' ' && *c && i < 11)
                _arg[i++] = *c++;

            arg[idx - 1] = strtoul(_arg, NULL, 0);
        }

        if (idx++ == 5)
            break;
    }

    l_cmd = idx - 1;

    list_for_each(pos, &boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if (strcmp(ent->cmd, _cmd) == 0)
            break;
    }

    if ((pos == &boot_shell_env.list) || (ent == NULL))
    {
        printf_e("\r\nUnkonw cmd: %s\n", _cmd);
        printf_e("\res#");
        return CMD_UNKONW;
    }

    if (l_cmd != ent->nr_arg)
    {
        printf_e("\r\nBad paramters, need %d paramters\n", ent->nr_arg);
        printf_e("\res#");
        return CMD_ARG_ERR;
    }

    printf_e("\n");

    switch (ent->nr_arg)
    {
        case 0:
            ((cfun_arg0)(ent->fun_cmd))();
            break;

        case 1:
            ((cfun_arg1)(ent->fun_cmd))(arg[0]);
            break;

        case 2:
            ((cfun_arg2)(ent->fun_cmd))(arg[0], arg[1]);
            break;

        case 3:
            ((cfun_arg3)(ent->fun_cmd))(arg[0], arg[1], arg[2]);
            break;

        case 4:
            ((cfun_arg4)(ent->fun_cmd))(arg[0], arg[1], arg[1], arg[3]);
            break;

        default:
            break;
    }

    if ((strncmp(_cmd, "exit", 16)))
    {
        shell_cmd_refresh(cmd);
        printf_e("\n\res#");
    }

    return CMD_SUCCESS;
}

/**
  * @brief  Check the password
  * @param  cmd: The password
  * @retval None
  */
static void shell_check_password(char *cmd)
{
    if ((strcmp(cmd, boot_shell_env.password)))
    {
        printf_e("\r\nPassword is incorrect\n");
        return;
    }

    boot_shell_env.check = 1;
    printf_e("\r\nes#");

    return;
}

/**
  * @brief  Handler for "UP" key
  * @retval None
  */
static void shell_up_func(void)
{
    uint8_t len   = 0;
    uint8_t l_cmd = boot_shell_env.cursor;

    switch (boot_shell_env.idx_cache)
    {
        case 0:
            if (boot_shell_env.cmd_cache0[0] == '\0')
                return;

            memcpy(boot_shell_env.cmd, boot_shell_env.cmd_cache0, 64);
            break;

        case 1:
            if (boot_shell_env.cmd_cache1[0] == '\0')
                return;

            memcpy(boot_shell_env.cmd, boot_shell_env.cmd_cache1, 64);
            break;

        case 2:
            if (boot_shell_env.cmd_cache2[0] == '\0')
                return;

            memcpy(boot_shell_env.cmd, boot_shell_env.cmd_cache2, 64);
            break;

        default:
            return;
    }

    while (boot_shell_env.cmd[len] != '\0')
        len++;

    if (l_cmd)
    {
        memset(boot_shell_env.out_buf, 0x0, 64);
        memset(boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", boot_shell_env.out_buf);
        memset(boot_shell_env.out_buf, ' ', l_cmd);
        printf_e("%s", boot_shell_env.out_buf);
        memset(boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", boot_shell_env.out_buf);
    }

    boot_shell_env.cursor = len;
    printf_e("%s", boot_shell_env.cmd);
    ++boot_shell_env.idx_cache;

    return;
}

/**
  * @brief  Handler for "DOWN" key
  * @retval None
  */
static void shell_down_func(void)
{
    uint8_t len   = 0;
    uint8_t l_cmd = boot_shell_env.cursor;

    switch (boot_shell_env.idx_cache)
    {
        case 3:
            if (boot_shell_env.cmd_cache1[0] == '\0')
                return;

            memcpy(boot_shell_env.cmd, boot_shell_env.cmd_cache1, 64);
            break;

        case 2:
            if (boot_shell_env.cmd_cache0[0] == '\0')
                return;

            memcpy(boot_shell_env.cmd, boot_shell_env.cmd_cache0, 64);
            break;

        case 1:
            memset(boot_shell_env.cmd, 0x0, 64);
            break;

        default:
            return;
    }

    while (boot_shell_env.cmd[len])
        ++len;

    if (l_cmd)
    {
        memset(boot_shell_env.out_buf, 0x0, 64);
        memset(boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e("%s", boot_shell_env.out_buf);
        memset(boot_shell_env.out_buf, ' ', l_cmd);
        printf_e(boot_shell_env.out_buf);
        memset(boot_shell_env.out_buf, 0x8, l_cmd);
        printf_e(boot_shell_env.out_buf);
    }

    boot_shell_env.cursor = len;
    ald_usart_send(boot_shell_env.h_usart, (uint8_t *)boot_shell_env.cmd, len, 100);
    --boot_shell_env.idx_cache;

    return;
}

/**
  * @brief  Handler for "TAB" key
  * @retval None
  */
static void shell_tab_func(void)
{
    uint8_t k, len = 0;
    uint32_t nr = 0;
    list_head_t *pos;
    boot_cmd_ent_t *ent, *_ent;

    if (!boot_shell_env.cursor)
    {
        printf_e("\r\n");

        list_for_each(pos, &boot_shell_env.list)
        {
            ent = list_entry(pos, boot_cmd_ent_t, link);

            printf_e("%s\t", ent->cmd);

            if (!(++nr & 0x3))
                printf_e("\r\n");
        }

        if (nr & 0x3)
            printf_e("\r\nes#");
        else
            printf_e("\res#");

        return;
    }

    list_for_each(pos, &boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strncmp(boot_shell_env.cmd, ent->cmd, boot_shell_env.cursor)) == 0)
        {
            if (pos->next != &boot_shell_env.list)
            {
                _ent = list_entry(pos->next, boot_cmd_ent_t, link);

                if ((strncmp(boot_shell_env.cmd, _ent->cmd, boot_shell_env.cursor)) != 0)
                    goto match_1;
                else
                    goto match_x;
            }
            else
            {
                goto match_1;
            }
        }
    }

    return;

match_1:

    for (k = boot_shell_env.cursor; ent->cmd[k]; ++k)
    {
        boot_shell_env.cmd[k] = ent->cmd[k];
        ++len;
    }

    ald_usart_send(boot_shell_env.h_usart, (uint8_t *)&boot_shell_env.cmd[boot_shell_env.cursor], len, 100);
    boot_shell_env.cursor = k;
    return;

match_x:
    printf_e("\r\n");

    list_for_each(pos, &boot_shell_env.list)
    {
        ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strncmp(boot_shell_env.cmd, ent->cmd, boot_shell_env.cursor)) == 0)
        {
            printf_e("%s\t", ent->cmd);

            if (!(++nr & 0x3))
                printf_e("\r\n");
        }
    }

    if (nr & 0x3)
        printf_e("\r\nes#");
    else
        printf_e("\res#");

    ald_usart_send(boot_shell_env.h_usart, (uint8_t *)boot_shell_env.cmd, boot_shell_env.cursor, 100);

    return;
}

/**
  * @brief  Receive a byte, This function is invoked in interrupt.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None
  */
static void shell_usart_rx_cbk(struct usart_handle_s *arg)
{
    boot_shell_env.rx_flag = 1;

    return;
}

/**
  * @}
  */

/** @defgroup Shell_Public_Functions Public Functions
  * @brief    Shell Public Functions
  * @{
  */
/**
  * @brief  Get receive flag
  * @retval 1-Set, 0-Reset.
  */
uint8_t shell_rx_flag_get(void)
{
    return boot_shell_env.rx_flag;
}

/**
  * @brief  Clear receive flag
  * @retval None
  */
void shell_rx_flag_clear(void)
{
    boot_shell_env.rx_flag = 0;
    return;
}

/**
  * @brief  Check timeout
  */
/**
  * @brief  Insert a command into list.
  * @param  cmd: Name of the command.
  * @param  func: Function of the command.
  * @param  nr_arg: Number of the command's function.
  * @retval Status:
  *           0 - Success
  *          -1 - Pointer to uart_handle_t is empty
  *          -2 - Memory is empty
  */
int32_t shell_cmd_insert(char *cmd, void *func, uint32_t nr_arg)
{
    list_head_t *pos;
    boot_cmd_ent_t *ent, *_ent;

    if (boot_shell_env.h_usart == NULL)
        return -1;

    ent = (boot_cmd_ent_t *)malloc(sizeof(boot_cmd_ent_t));

    if (ent == NULL)
        return -2;

    memset(ent, 0x0, sizeof(boot_cmd_ent_t));
    ent->fun_cmd = func;
    ent->nr_arg = nr_arg;
    memcpy(ent->cmd, cmd, 15);

    list_for_each(pos, &boot_shell_env.list)
    {
        _ent = list_entry(pos, boot_cmd_ent_t, link);

        if ((strcmp(ent->cmd, _ent->cmd)) < 0)
            break;
    }

    list_add_tail(&ent->link, pos);
    ++boot_shell_env.nr_cmd;

    return 0;
}

/**
  * @brief  Initialize shell module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void shell_init(usart_handle_t *hperh)
{
    if (hperh == NULL)
    {
        boot_shell_env.h_usart = hperh;
        return;
    }

    memset(&boot_shell_env, 0x0, sizeof(boot_shell_env));
    INIT_LIST_HEAD(&boot_shell_env.list);
    boot_shell_env.h_usart = hperh;
    boot_shell_env.h_usart->rx_cplt_cbk = shell_usart_rx_cbk;
    strcpy(boot_shell_env.password, "eastsoft");
    boot_shell_env.check = SET;

    printf_e("\r\n");
    printf_e("*************************************\r\n");
    printf_e("*        Welcome to Eastsoft        *\r\n");
    printf_e("*************************************\r\n\r\n");

    ald_usart_recv_by_it(boot_shell_env.h_usart, (uint8_t *)&boot_shell_env.c_buf, 1);

    return;
}

/**
  * @brief  Function for shell's task.
  * @param  arg: The parameter.
  * @retval None
  */
void shell_task_func(void *arg)
{
    char c;

    c = boot_shell_env.c_buf;

    if (c == 0x1B)  /*ESC*/
    {
        boot_shell_env.fn_key[0] = 0x1B;
        ald_usart_recv_by_it(boot_shell_env.h_usart, (uint8_t *)&boot_shell_env.c_buf, 1);
        return;
    }

    if (boot_shell_env.fn_key[0] == 0x1B)   /* Up and Down */
    {
        if (c == '[')
        {
            boot_shell_env.fn_key[1] = c;
        }
        else if (c == 'A')
        {
            shell_up_func();
            memset(boot_shell_env.fn_key, 0x0, 4);
        }
        else if (c == 'B')
        {
            shell_down_func();
            memset(boot_shell_env.fn_key, 0x0, 4);
        }
        else
        {
            boot_shell_env.fn_key[0] = 0;
        }

        ald_usart_recv_by_it(boot_shell_env.h_usart, (uint8_t *)&boot_shell_env.c_buf, 1);
        return;
    }

    if ((c == ' ' || (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')
            || (c >= '0' && c <= '9') || (c == '-')
            || (c == '_')) && (boot_shell_env.cursor < 64))
    {
        boot_shell_env.cmd[boot_shell_env.cursor++] = c;

        if (boot_shell_env.check)
            ald_usart_send(boot_shell_env.h_usart, (uint8_t *)&c, 1, 100);
    }

    switch (c)
    {
        case 0x08:  /* Backspace */
            if (!boot_shell_env.cursor)
                break;

            boot_shell_env.cmd[--boot_shell_env.cursor] = 0;

            if (!boot_shell_env.check)
                break;

            ald_usart_send(boot_shell_env.h_usart, (uint8_t *)&c, 1, 100);

            break;

        case 0x09:  /* TAB */
            shell_tab_func();
            break;

        case 0x0D:  /* Enter */
            if (!boot_shell_env.check)
            {
                shell_check_password(boot_shell_env.cmd);
                memset(boot_shell_env.cmd, 0x0, 64);
                boot_shell_env.cursor = 0;
                break;
            }

            shell_cmd_exec(boot_shell_env.cmd);
            memset(boot_shell_env.cmd, 0x0, 64);
            boot_shell_env.idx_cache = 0;
            boot_shell_env.cursor    = 0;
            break;

        default:
            break;
    }

    ald_usart_recv_by_it(boot_shell_env.h_usart, (uint8_t *)&boot_shell_env.c_buf, 1);
    return;
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(uart_buf, 64, fmt, args);
    va_end(args);

    ald_usart_send(boot_shell_env.h_usart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
    ald_delay_ms(1);
    return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
