/**
  *********************************************************************************
  * @file   boot.c
  * @brief  Base functions.
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include <stdio.h>
#include "ald_syscfg.h"
#include "ald_rmu.h"
#include "ald_usart.h"
#include "boot.h"
#include "boot_flash.h"
#include "boot_frame.h"
#include "boot_shell.h"
#include "boot_crc.h"


/** @defgroup Bootloader Bootloader
  * @brief    Eastsoft Bootloader
  * @{
  */

/** @defgroup Base Base
  * @brief    Bootloader Base
  * @{
  */
/** @defgroup Base_Public_Variables Public Variable
  * @brief    Base Public Variable
  * @{
  */
timer_env_t timer_env = {0};
uint32_t __crc;
int32_t __len;
uint32_t __bufb[256];
/**
  * @}
  */

/** @defgroup Base_Private_Functions Private Functions
  * @brief    Base Private_Functions
  * @{
  */
/**
  * @brief  Systick callback function
  * @retval None
  */
__isr__ void ald_systick_irq_cbk(void)
{
    if (!timer_env.state)
        return;

    if ((--timer_env.tick == 0) && (timer_env.cbk != NULL))
        timer_env.cbk();
}

/**
  * @brief  Initialize timer.
  * @retval None
  */
void timer_init(void)
{
    memset(&timer_env, 0x0, sizeof(timer_env_t));
    return;
}

/**
  * @brief  Start timer.
  * @param  tick: Timeout in milliseconds.
  * @param  cbk: callback function for timeout.
  * @retval None
  */
void timer_start(uint32_t tick, timer_cbk cbk)
{
    tick = tick == 0 ? 1 : tick;

    __disable_irq();
    timer_env.tick  = tick;
    timer_env.cbk   = cbk;
    timer_env.state = 1;
    __enable_irq();

    return;
}

/**
  * @brief  Stop timer.
  * @retval None
  */
void timer_stop(void)
{
    __disable_irq();
    memset(&timer_env, 0x0, sizeof(timer_env_t));
    __enable_irq();

    return;
}

/**
  * @brief  Callback function for timeout.
  * @retval None
  */
void boot_timer_cbk(void)
{
    timer_env.timeout = 1;
}

/**
  * @brief  Run in flash
  * @retval None.
  */
void run_in_flash(void)
{
    uint32_t addr = 0;

    ald_usart_reset(frame_env.h_usart);
    ald_crc_reset(&h_crc);

    addr = *(uint32_t *)(APP_FLASH_S_0 + 4);
    ald_vtor_config(OFFSET_VTOR_FLASH, ENABLE);
	__set_MSP(*(uint32_t *)APP_FLASH_S_0);
    ((void(*)(void))(addr))();

    return;
}

/**
  * @brief  command for "version".
  * @retval None
  */
void cmd_version(void)
{
    printf_e("\rBootload: 01-00-00-00\r\n");
    return;
}

/**
  * @brief  command for "ckeck".
  * @retval None
  */
void cmd_ckeck(uint32_t crc, uint32_t len)
{
	uint32_t _crc;

	printf_e("\rcrc: %u, 0x%X\r\n", crc, crc);
	printf_e("\rlen: %d\r\n", len);

	if (len > (APP_FLASH_E_1 - APP_FLASH_S_1)) {
		printf_e("\rlen is too large!\r\n");
		return;
	}

	_crc = crc_digest((unsigned char *)(APP_FLASH_S_1), len);
	printf_e("\rImage crc: 0x%X\r\n", _crc);
	
	if (_crc == crc) {
		__crc = crc;
		__len = len;
		printf_e("\rCheck: PASS!\r\n", _crc);
	}
	else {
		printf_e("\rCheck: ERROR!\r\n", _crc);
	}

	return;
}

/**
  * @brief  command for "move".
  * @retval None
  */
void cmd_move(void)
{
	uint32_t _crc, addr0, addr1;
	int i, k, len;

	addr0 = APP_FLASH_S_0;

	while (addr0 < APP_FLASH_E_0) {
		ald_iap_erase_page(addr0, 1);
		addr0 += 2048;
	}

	len   = 0;
	addr0 = APP_FLASH_S_0;
	addr1 = APP_FLASH_S_1;

	for (i = 0; len < __len; ++i) {
		for (k = 0; k < 256; ++k) {
			__bufb[k] = *(uint32_t *)(addr1 + (k << 2));
		}
		
		ald_iap_program_words(addr0, (uint8_t *)__bufb, 1024, ENABLE);

		len   += 1024;
		addr0 += 1024;
		addr1 += 1024;
	}

	_crc = crc_digest((unsigned char *)(APP_FLASH_S_0), __len);

	if (_crc == __crc) {
		printf_e("\rCRC: 0x%X - 0x%X\r\n", __crc, _crc);
		printf_e("\rLEN: %d\r\n", __len);
		printf_e("\rMove success!\r\n", _crc);
	}
	else {
		printf_e("\rMove failed!\r\n", _crc);
	}

	return;
}


/**
  * @brief  command for "reboot".
  * @retval None
  */
void cmd_reboot(void)
{
    ald_rmu_reset_periperal(RMU_PERH_CHIP);
    return;
}

static void usart1_irq_ctrl(type_func_t status)
{
    ald_mcu_irq_config(USART1_IRQn, 3, status);
    return;
}

/**
  * @brief  command for "download_ram".
  * @retval None
  */
void cmd_download(void)
{
    usart1_irq_ctrl(DISABLE);
    printf_e("\r\n************* Update Program *************");
    printf_e("\r\nStrat updating afer 3 second !");
    printf_e("\r\nDon't input anything!!!");
    ald_delay_ms(3000);

    flash_usr_page_erase();
    boot_image_update();
    ald_delay_ms(100);
    boot_update_info();
    usart1_irq_ctrl(ENABLE);

    return;
}

/**
  * @brief  command for "run_flash".
  * @retval None
  */
void run_flash_cmd(void)
{
    if (!flash_usr_page_get())
    {
        printf_e("\rThe APP flash is empty, Please Update first!\r\n");
        return;
    }

    printf_e("\r\n");
    run_in_flash();
    return;
}

/**
  * @}
  */
/** @defgroup Base_Public_Functions Public functions
  * @brief    Base Public Functions
  * @{
  */
/**
  * @brief  Enter bootloader.
  * @retval None
  */
void boot_enter(usart_handle_t *hperh)
{
    timer_init();
    boot_frame_init(hperh);
    shell_init(hperh);
    shell_cmd_insert("version", (void *)cmd_version, 0);
    shell_cmd_insert("reboot", (void *)cmd_reboot, 0);
    shell_cmd_insert("check", (void *)cmd_ckeck, 2);
    shell_cmd_insert("move", (void *)cmd_move, 0);
    shell_cmd_insert("update", (void *)cmd_download, 0);
    shell_cmd_insert("run", (void *)run_flash_cmd, 0);

    if (!flash_usr_page_get())
    {
        printf_e("The APP flash is empty, Please Update first!\r\n");
    }
    else
    {
        printf_e("The APP flash is unempty, You can type the following cmd:\r\n");
        printf_e("update, run.\r\n");
        printf_e("Otherwise auto run after 3 seconds:\r\n");
        timer_start(3000, boot_timer_cbk);
    }

    printf_e("\r\nes#");

    while (1)
    {
        if (shell_rx_flag_get())
        {
            shell_rx_flag_clear();
            shell_task_func(NULL);

            if (timer_env.state)
                timer_stop();
        }

        if (timer_env.timeout)
        {
            timer_env.timeout = 0;
            run_in_flash();
		}
    }
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
