/**********************************************************************************
 *
 * @file    exec_proc.c
 * @brief   define the exection functions for state machine
 *
 * @date    22 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Feb  2023    shicc           version:1.0.1
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "exec_proc.h"
#include "main.h"


/* Private Macros ------------------------------------------------------------ */
typedef  void (*FunVoidType)(void);

#if (APP_ADDR & 0x03)
    #error " app address not aligned"
#endif /*  */

/* Public Variables ---------------------------------------------------------- */


/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */


/* Public function prototypes ----------------------------------------------- */


/* Private Variables--------------------------------------------------------- */

/* Public Function ---------------------------------------------------------- */


/* Private Function ---------------------------------------------------------- */

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    SYSCFG_UNLOCK();
    md_rmu_enable_gpio_reset();
    md_rmu_enable_uart0_reset();
    SYSCFG_LOCK();
}

/**
  * @brief  define the page erase function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_page_erase(uint32_t PageCode)
{

    volatile fsm_iap_result_t status = IAP_SUCCESS;

    /* size of each flash page is 512byte */
    uint32_t address = (PageCode << 9);

    if (address & 0x3)
        return IAP_FAIL;

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    if (IAP_SUCCESS != IAPROM_PAGE_ERASE(address, 0))
        status = IAP_FAIL;

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    return status;
}

/**
  * @brief  define the words program function.
  * @param  None
  * @retval None
  */
static fsm_iap_result_t fsm_words_program(uint32_t address, uint32_t data_address, uint32_t len)
{
    volatile fsm_iap_result_t status = IAP_SUCCESS;

    if (address & 0x3)
        return IAP_FAIL;

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    if (IAP_SUCCESS != IAPROM_MULTI_PROGRAM(address, data_address, len, AUTO_ERASE_FALSE))
        status = IAP_FAIL;

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    return status;
}

/**
  * @brief  define Reverse8 function.
  * @param  None
  * @retval None
  */

uint8_t     Reverse8(uint8_t U8)
{
    U8 = (((U8 & 0xaa) >> 1) | ((U8 & 0x55) << 1));
    U8 = (((U8 & 0xcc) >> 2) | ((U8 & 0x33) << 2));
    U8 = (((U8 & 0xf0) >> 4) | ((U8 & 0x0f) << 4));
    return (U8);
}

/**
  * @brief  define Reverse32 function, used to crc check.
  * @param  None
  * @retval None
  */

uint32_t    Reverse32(uint32_t U32)
{
    U32 = (((U32 & 0xaaaaaaaa) >> 1) | ((U32 & 0x55555555) << 1));
    U32 = (((U32 & 0xcccccccc) >> 2) | ((U32 & 0x33333333) << 2));
    U32 = (((U32 & 0xf0f0f0f0) >> 4) | ((U32 & 0x0f0f0f0f) << 4));
    U32 = (((U32 & 0xff00ff00) >> 8) | ((U32 & 0x00ff00ff) << 8));
    U32 = (((U32 & 0xffff0000) >> 16) | ((U32 & 0x0000ffff) << 16));
    return (U32);
}

/**
  * @brief  define the checksum function.
  * @param  None
  * @retval None
  */
static uint32_t fsm_get_crc32(uint32_t *data_ptr, uint32_t len)
{
    uint32_t    crc32, poly, data32;
    uint8_t     data8;
    uint8_t     ii;
    uint32_t    *pU32Buf;

    __disable_irq();
    MD_BOOT_FROM_FLASH();

    pU32Buf = (uint32_t *)((uint32_t)data_ptr + APP_ADDR);
//    pU32Buf = (uint32_t *)((uint32_t)data_ptr);
    crc32 = 0xffffffffUL;
    poly = 0x04c11db7;
    len >>= 2;

    while (len--)
    {
        data32 = *pU32Buf++;

        for (ii = 0 ; ii < 4 ; ii++)
        {
            data8 = data32 & 0xff;
            data8 = Reverse8(data8);
            data32 >>= 8;
            data8 ^= (crc32 >> 24);
            crc32 <<= 8;

            if (data8 & 0x80) crc32 ^= 0x690CE0EE; // (poly << 7) ^ (poly << 1)

            if (data8 & 0x40) crc32 ^= 0x34867077; // (poly << 6) ^ (poly << 0)

            if (data8 & 0x20) crc32 ^= 0x9823B6E0; // (poly << 5)

            if (data8 & 0x10) crc32 ^= 0x4C11DB70; // (poly << 4)

            if (data8 & 0x08) crc32 ^= 0x2608EDB8; // (poly << 3)

            if (data8 & 0x04) crc32 ^= 0x130476DC; // (poly << 2)

            if (data8 & 0x02) crc32 ^= 0x09823B6E; // (poly << 1)

            if (data8 & 0x01) crc32 ^= 0x04C11DB7; // (poly << 0)
        }
    }

    crc32 = ~Reverse32(crc32);

    MD_BOOT_FROM_BOOT_FLASH();
    __enable_irq();

    return crc32;
}

/**
  * @brief  define the check empty function.
  * @param  None
  * @retval None
  */
static fsm_result_t fsm_check_empty(uint32_t *data_ptr, uint32_t len)
{
    uint32_t  i;

    if ((((uint32_t)data_ptr) % 4 != 0) || (len % 4 != 0))
    {
        return FAIL;
    }

    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    for (i = 0; i < len; i += 4)
    {
        if (*(data_ptr++) != 0xFFFFFFFF)
        {
            break;
        }
    }

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    if (i < len)
        return FAIL;
    else
        return PASS;
}

/**
  * @brief  define the memory read function.
  * @param  None
  * @retval None
  */
static void fsm_read_memo(uint8_t *dest, void *src, uint32_t len)
{
    __disable_irq();

    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    memcpy(dest, (const void *)src, len);

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

    __enable_irq();

    return;
}

#ifdef  __USE_CRC32
/**
  * @brief  init crc32.
  * @param  None
  * @retval None
  */
static uint8_t init_crc32(void)
{
    md_crc_init_t crc_init;

    /* Clear crc_handle_t structure */
    memset(&crc_init, 0x0, sizeof(crc_init));
    /* Initialize CRC */
    md_crc_init_struct(&crc_init);
    crc_init.mode = MD_CRC_MODE_32;
    crc_init.len = MD_CRC_DATASIZE_32;
    crc_init.order = MD_CRC_BYTORD_LOW;
    crc_init.seed = 0xFFFFFFFF;
    crc_init.chs_inv = ENABLE;
    crc_init.chs_rev = ENABLE;
    crc_init.data_inv = DISABLE;
    crc_init.data_rev = ENABLE;
    md_crc_init(&crc_init);

    return 0;
}


/**
  * @brief  check specified checksum function.
  * @param  None
  * @retval None
  */
uint8_t verify_valid(void)
{
    uint32_t  len_hex, crc_cal_result, crc_hex_result;
    uint32_t i;
    uint32_t *data_ptr = (uint32_t *)APP_ADDR;

    if (!md_crc_is_enabled(CRC))
        init_crc32();

    if (CRC_CAL_ADDR & 0x03)
        return FAIL;

    /* get info from specified address */
    crc_hex_result = *((uint32_t *)CRC_CAL_ADDR);
    len_hex = *((uint32_t *)CRC_CAL_ADDR + 1);

    if (len_hex > 0x7E000 || len_hex & 0x03 || len_hex == 0)
        return FAIL;

    __disable_irq();
    SYSCFG_UNLOCK();
    /* set to app flash*/
    CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();

#ifdef __IAR_SYSTEMS_ICC__  /* IAR Compiler */
    /*===========================================================================================*/
    /* MISRA violation of rule 11.4 17.4 -pointer arithmetic is used for CRC calculation */
#pragma diag_suppress=Pm088,Pm141
#endif  /* __IAR_SYSTEMS_ICC__ */

    for (i = 0; i < len_hex / 4; i++)
        md_crc_write_data(CRC, *data_ptr++);

    crc_cal_result = md_crc_get_check_result(CRC);

#ifdef __IAR_SYSTEMS_ICC__  /* IAR Compiler */
    /*===========================================================================================*/
    /* MISRA violation of rule 11.4 17.4 -pointer arithmetic is used for CRC calculation */
#pragma diag_suppress=Pm088,Pm141
#endif  /* __IAR_SYSTEMS_ICC__ */

    SYSCFG_UNLOCK();
    /* set to boot flash*/
    SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK);
    SYSCFG_LOCK();
    __enable_irq();

    if (crc_hex_result == crc_cal_result)
    {
        return PASS;
    }
    else
    {
        MD_CRC_RESET();
        MD_CRC_DISABLE();
        return FAIL;
    }
}
#endif
/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    FunVoidType JumpToApplication = NULL;
    uint32_t m_JumpAddress;
    uint32_t addr;

    __disable_irq();

    if (para == GO_APP)
    {
        addr = APP_ADDR;
    }
    else if (para == GO_BOOT)
    {
        addr = BOOT_ADDR;
    }

    /* reset registers of peripherals */
    sfr_reset();

    /* disable all peripherals' clock */
    SYSCFG_UNLOCK();
    md_cmu_disable_perh_all();
    SYSCFG_LOCK();

    /* disable all peripherals which may cause an interrupt,and clear all possible undisposed interrupt flag */
    NVIC->ICER[0] = 0xFFFFFFFF;
    NVIC->ICER[1] = 0xFFFFFFFF;
    NVIC->ICER[2] = 0xFFFFFFFF;
    NVIC->ICER[3] = 0xFFFFFFFF;
    NVIC->ICER[4] = 0xFFFFFFFF;
    NVIC->ICER[5] = 0xFFFFFFFF;
    NVIC->ICER[6] = 0xFFFFFFFF;
    NVIC->ICER[7] = 0xFFFFFFFF;

    NVIC->ICPR[0] = 0xFFFFFFFF;
    NVIC->ICPR[1] = 0xFFFFFFFF;
    NVIC->ICPR[2] = 0xFFFFFFFF;
    NVIC->ICPR[3] = 0xFFFFFFFF;
    NVIC->ICPR[4] = 0xFFFFFFFF;
    NVIC->ICPR[5] = 0xFFFFFFFF;
    NVIC->ICPR[6] = 0xFFFFFFFF;
    NVIC->ICPR[7] = 0xFFFFFFFF;

    /* disable systick and clear the pending bit */
    SysTick->CTRL = 0;
    SCB->ICSR |= SCB_ICSR_PENDSTCLR_Msk;

    /* set start adress to app/boot flash*/
    SYSCFG_UNLOCK();

    if (para == GO_APP)
    {
        SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    }
    else if (para == GO_BOOT)
    {
        SYSCFG_BOOTFLASH_MAPPING_ENABLE();
    }

    /* interrupt vector remap */
    SYSCFG->VTOR = addr & 0xFFFFFF00;

    SYSCFG_LOCK();

    __enable_irq();

    m_JumpAddress = *(volatile uint32_t *)((addr & 0xFFFFFF00) + 4);
    JumpToApplication = (FunVoidType) m_JumpAddress;

    /* init stack top */
    __set_MSP(*(volatile uint32_t *)(addr & 0xFFFFFF00));

    /* jump to app/boot flash */
    JumpToApplication();

}

/**
  * @brief  assign function pointer to all the state machine subfunction.
  * @param  None
  * @retval None
  */
void fsm_exec_func_init(void)
{
    g_isp_data.p_page_erase   = &fsm_page_erase;
    g_isp_data.p_words_progrm = &fsm_words_program;
    g_isp_data.p_get_crc32    = &fsm_get_crc32;
    g_isp_data.p_check_empty  = &fsm_check_empty;
    g_isp_data.p_go           = &fsm_go;
    g_isp_data.p_read_memo    = &fsm_read_memo;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
