/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    06 Jun. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          06 Jun. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
md_wwdt_inittypedef g_WWDT_Init;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    md_mcu_irq_config(WWDT_IRQn, 3, ENABLE);
    __enable_irq();

    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_wwdt(RCU);

    uart_pin_init();
    uart_init();

    printf("MCU reset\r\n");
    printf("WWDT demo\r\n");

    /* Initialize WWDT */
    memset(&g_WWDT_Init, 0x0, sizeof(md_wwdt_inittypedef));
    g_WWDT_Init.Prescaler = WWDT_PRESCALER_8;
    g_WWDT_Init.Window    = 0x60;
    g_WWDT_Init.Counter   = 0x7F;
    g_WWDT_Init.EWIMode   = WWDT_EWI_ENABLE;
    md_wwdt_init(WWDT, &g_WWDT_Init);

    while (1)
    {
        md_delay_1ms(1000);
    }
}

int fputc(int c, FILE *f)
{
    while (!(UARTX->STAT & MD_UART_STATUS_TFEMPTY));

    UARTX->TXBUF = (uint8_t)c;

    return c;
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
