/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    08 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          08 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

char g_tx_buf[512];

uint8_t g_rx_buf[10];
uint8_t g_rx_i = 0U;

uint32_t g_data_addr;
uint8_t g_data_len;

uint8_t g_frame_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    /* Initialize rx pin */
    gpio_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpio_init);
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void init_uart(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);

    md_uart_set_rec_timeout(UARTX, 0xFF);
    md_uart_enable_rtoen(UARTX);

    md_uart_enable_it_rfnempty(UARTX);
}

uint8_t ask_data_handle(void)
{
    if (g_rx_buf[0] != STA1 || g_rx_buf[1] != STA2 || g_rx_buf[2] != FRAME_LEN
            || g_rx_buf[8] != END1 || g_rx_buf[9] != END2)
        return 0;

    g_data_addr = (g_rx_buf[3] << 24) | (g_rx_buf[4] << 16) | (g_rx_buf[5] << 8) | (g_rx_buf[6]);
    g_data_len = g_rx_buf[7];

    return 1;
}

void answer_data_send(void)
{
    uint16_t i;
    uint8_t data_buf[512];

    memcpy(data_buf, (void *)g_data_addr, g_data_len);

    g_tx_buf[0] = STA1;
    g_tx_buf[1] = STA2;
    g_tx_buf[2] = FRAME_LEN + g_data_len;
    g_tx_buf[3] = g_rx_buf[3];
    g_tx_buf[4] = g_rx_buf[4];
    g_tx_buf[5] = g_rx_buf[5];
    g_tx_buf[6] = g_rx_buf[6];
    g_tx_buf[7] = g_data_len;

    for (i = 0; i < g_data_len; i++)
        g_tx_buf[8 + i] = data_buf[i];

    g_tx_buf[8 + g_data_len] = END1;
    g_tx_buf[9 + g_data_len] = END2;

    md_uart_enable_it_tfempty(UARTX);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    md_mcu_irq_config(UART1_IRQn, 2, ENABLE);
    __enable_irq();

    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);

    uart_pin_init();

    init_uart();

    while (1)
    {
        if (g_frame_flag == 1)
        {
            if (ask_data_handle() == 1)
                answer_data_send();

            memset(g_rx_buf, 0x00, FRAME_LEN);

            g_frame_flag = 0;
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
