/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    26 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize GPIO
  * @param  None
  * @retval None
  */
void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.odrv = MD_GPIO_DRIVING_8MA;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_5;
    md_gpio_init(CH1_PORT, CH1_PIN, &gpio_init);
    md_gpio_init(CH1N_PORT, CH1N_PIN, &gpio_init);
    md_gpio_init(CH2_PORT, CH2_PIN, &gpio_init);
    md_gpio_init(CH2N_PORT, CH2N_PIN, &gpio_init);
    md_gpio_init(CH3_PORT, CH3_PIN, &gpio_init);
    md_gpio_init(CH3N_PORT, CH3N_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C6T
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c6t_init;
    md_timer_oc_init_t oc_init;

    md_timer_base_struct_init(&ad16c6t_init);        /* initialize the ad16c6t_init  */
    ad16c6t_init.prescaler = 72 - 1;                 /* clk_count: 72 */
    ad16c6t_init.clk_div = MD_TIMER_CLOCK_DIV1;      /* working clock of dead time and filter */
    ad16c6t_init.mode = MD_TIMER_CNT_MODE_CENTER1;   /* count up */
    ad16c6t_init.period = 5000 - 1;                  /* period is 5000 count clock */
    ad16c6t_init.re_cnt = 1 - 1;                     /* 0 repeat count */
    md_timer_base_set_config(AD16C6T1, &ad16c6t_init);

    /* Timer output compare1/2/3 preload enable */
    md_timer_enable_output_compare1_preload_ch1pen(AD16C6T1);
    md_timer_enable_output_compare2_preload_ch2pen(AD16C6T1);
    md_timer_enable_output_compare3_preload_ch3pen(AD16C6T1);

    md_timer_oc_struct_init(&oc_init);
    oc_init.ocstate = MD_TIMER_OCSTATE_ENABLE;    /* enable CHxO compare output */
    oc_init.ocnstate = MD_TIMER_OCSTATE_ENABLE;   /* enable CHxON compare output */
    oc_init.oc_mode = MD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    oc_init.oc_polarity = MD_TIMER_OC_POLARITY_HIGH;    /* CHxO compare output polarity is high */
    oc_init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;  /* CHxON compare output polarity is high */
    oc_init.oc_idle = MD_TIMER_OC_IDLE_RESET;           /* CHxO idle is low */
    oc_init.ocn_idle = MD_TIMER_OCN_IDLE_RESET;         /* CHxON idle is low */
    oc_init.pulse = 1500 - 1;    /* CH1 compare value is 1500 - 1 */
    md_timer_oc_init(AD16C6T1, MD_TIMER_CHANNEL_1, &oc_init);
    oc_init.pulse = 2500 - 1;    /* CH2 compare value is 2500 - 1 */
    md_timer_oc_init(AD16C6T1, MD_TIMER_CHANNEL_2, &oc_init);
    oc_init.pulse = 3500 - 1;    /* CH3 compare value is 3500 - 1 */
    md_timer_oc_init(AD16C6T1, MD_TIMER_CHANNEL_3, &oc_init);

    md_timer_enable_it_ch1(AD16C6T1);   /* enable CC1 interrupt request */
    md_timer_enable_it_ch2(AD16C6T1);   /* enable CC2 interrupt request */
    md_timer_enable_it_ch3(AD16C6T1);   /* enable CC3 interrupt request */
    md_timer_enable_it_upd(AD16C6T1);   /* update interrupt request */

    md_timer_enable_main_output_goen(AD16C6T1);  /* enable output */
    md_timer_enable_counter_cnten(AD16C6T1);     /* enable AD16C6T1 */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_mcu_irq_config(AD16C6T1_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_ad16c6t1(RCU);

    /* Initialize pin */
    init_gpio();
    /* Initialize timer */
    init_timer();

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
