/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    26 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          26 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uint16_t g_duty_buffer[BUFFER_SIZE];
uint16_t g_cycle_buffer[BUFFER_SIZE];

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize IO
  * @param  None
  * @retval None
  */
void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.odrv = MD_GPIO_DRIVING_8MA;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_6;
    md_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C6T
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c6t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&ad16c6t_init);        /* initialize the ad16c6t_init  */
    ad16c6t_init.prescaler = 72 - 1;                 /* clk_count: 1MHz */
    ad16c6t_init.clk_div = MD_TIMER_CLOCK_DIV1;      /* working clock of dead time and filter */
    ad16c6t_init.mode = MD_TIMER_CNT_MODE_UP;        /* count up */
    ad16c6t_init.period = 0xFFFF;                    /* set max period */
    ad16c6t_init.re_cnt = 1 - 1;                     /* 0 repeat count */
    md_timer_base_set_config(AD16C6T1, &ad16c6t_init);

    md_timer_ic_struct_init(&ic_init);                 /* initialize the ic_init  */
    ic_init.filter      = 0;                           /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;   /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;      /* capture this channel */
    md_timer_ic_init(AD16C6T1, MD_TIMER_CHANNEL_1, &ic_init);

    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;   /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;    /* capture adjacent channel */
    md_timer_ic_init(AD16C6T1, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(AD16C6T1, MD_TIMER_SLAVEMODE_RESET);        /* Reset Mode */
    md_timer_set_slave_trigger_tssel(AD16C6T1, 5);     /* 0x5: Filtered Timer Input 1 */

    md_timer_set_cc_dma_select_ccdmasel(AD16C6T1, MD_TIMER_DMASELECTION_COMPARE);  /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_dma_ch1(AD16C6T1);                  /* enable CC1 DMA request */
    md_timer_enable_dma_ch2(AD16C6T1);                  /* enable CC2 DMA request */

    md_timer_enable_it_upd(AD16C6T1);                  /* enable updata interrupt */
    md_timer_enable_it_ch1(AD16C6T1);         /* enable CC1 interrupt request */
    md_timer_enable_it_ch2(AD16C6T1);         /* enable CC1 interrupt request */

    /* enable AD16C6T1 */
    md_timer_enable_counter_cnten(AD16C6T1);
}

/**
  * @brief:  Use MD library to config DMA.
  * @param:  None
  * @retval: None
  */
static void init_dma(void)
{
    md_dma_config_t dma_config_cycle;
    md_dma_config_t dma_config_duty;

    memset(&dma_config_cycle, 0x0, sizeof(md_dma_config_t));
    dma_config_cycle.src            = (void *)(&AD16C6T1->CCVAL1);
    dma_config_cycle.dst            = (void *)g_cycle_buffer;
    dma_config_cycle.size           = BUFFER_SIZE;
    dma_config_cycle.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_cycle.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_cycle.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config_cycle.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config_cycle.R_power        = MD_DMA_R_POWER_1;
    dma_config_cycle.priority       = MD_DMA_LOW_PRIORITY;
    dma_config_cycle.mem_to_mem     = DISABLE;
    dma_config_cycle.circle_mode    = ENABLE;
    dma_config_cycle.msigsel        = MD_DMA_MSIGSEL_AD16C6T1_CH1;
    dma_config_cycle.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA, MD_DMA_CH_0, &dma_config_cycle);

    memset(&dma_config_duty, 0x0, sizeof(md_dma_config_t));
    dma_config_duty.src            = (void *)(&AD16C6T1->CCVAL2);
    dma_config_duty.dst            = (void *)g_duty_buffer;
    dma_config_duty.size           = BUFFER_SIZE;
    dma_config_duty.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_duty.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config_duty.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config_duty.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config_duty.R_power        = MD_DMA_R_POWER_1;
    dma_config_duty.priority       = MD_DMA_LOW_PRIORITY;
    dma_config_duty.mem_to_mem     = DISABLE;
    dma_config_duty.circle_mode    = ENABLE;
    dma_config_duty.msigsel        = MD_DMA_MSIGSEL_AD16C6T1_CH2;
    dma_config_duty.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA, MD_DMA_CH_1, &dma_config_duty);

    md_dma_enable_it_tc(DMA, MD_DMA_CH_0);
    md_dma_enable_it_tc(DMA, MD_DMA_CH_1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_mcu_irq_config(AD16C6T1_IRQn, 3, ENABLE);
    md_mcu_irq_config(DMA1_CH0_IRQn, 3, ENABLE);
    md_mcu_irq_config(DMA1_CH1_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_dmamux(RCU);
    md_rcu_enable_ad16c6t1(RCU);

    /* Initialize pin */
    init_gpio();

    /* Initialize dma */
    init_dma();

    /* Initialize timer */
    init_timer();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
