/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    03 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          03 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint32_t g_intr_wakeup;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private functions ----------------------------------------------------------*/

/**
  * @brief  Initializate pins.
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize All pins analog mode */
    md_gpio_init_struct(&gpio_init);
    gpio_init.mode  = MD_GPIO_MODE_ANALOG;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(GPIOA, MD_GPIO_PIN_0, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_1, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_2, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_3, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_4, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_5, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_6, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_7, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_8, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_9, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_10, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_11, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_12, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_13, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_14, &gpio_init);
    md_gpio_init(GPIOA, MD_GPIO_PIN_15, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_0, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_1, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_2, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_3, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_4, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_5, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_6, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_7, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_8, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_9, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &gpio_init);
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_0, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_1, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_2, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_3, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_4, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_5, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_6, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_7, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_8, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_9, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_10, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_11, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_12, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_13, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_14, &gpio_init);
    md_gpio_init(GPIOC, MD_GPIO_PIN_15, &gpio_init);

    /* Initialize LED pin */
    md_gpio_init_struct(&gpio_init);
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    md_gpio_set_pin_high(LED_PORT, LED_PIN);

    /* Initialize KEY pin */
    md_gpio_init_struct(&gpio_init);
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(KEY_PORT, KEY_PIN, &gpio_init);

    return;
}

/**
  * @brief  EnterSleep
  * @note   Enable sleep mode
  * @param  Mode
  * @retval None
  */
void EnterSleep(uint8_t  mode)
{
    /* Ensure we SLEEP - SLEEPDEEP should be clear */

    uint32_t systick_en = 0U;

    systick_en = SysTick->CTRL & SysTick_CTRL_ENABLE_Msk;
    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;

    if (mode)       /* SCR[2] = 1 (SLEEPDEEP) */
        SCB->SCR |= (1UL << 2);
    else            /* SCR[2] = 0 (SLEEP) */
        SCB->SCR &= ~(1UL << 2);

    __WFI();    /* Wait For Interrupt */

    SysTick->CTRL |= systick_en;
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    md_mcu_irq_config(WAKEUP_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_gpiob(RCU);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    init_gpio();

    md_exti_set_rts(EXTI, MD_EXTI_EVENT_WAKEUP);
    md_exti_enable_it_wakeup(EXTI);

    md_syscfg_set_lowpower_mode(SYSCFG, MD_SYSCFG_LPLS_STOP_MODE);
    md_syscfg_enable_wakeup_edge_rising(SYSCFG, MD_SYSCFG_WKEG_WKUP4);
    md_syscfg_enable_wakeup(SYSCFG, MD_SYSCFG_WKEN_WKUP4);
    md_fc_enable_flash_stop_mode(FC);   /* Flash: open stop mode,The current can be reduced about 20uA */

    EnterSleep(1);

    while (!g_intr_wakeup);

    md_rcu_clear_flag_reset_flag(RCU);  /* Clear reset flag */

    while (1)
    {
        md_delay_1ms(250);

        if (md_gpio_get_outputpinset(LED_PORT, LED_PIN))
            md_gpio_set_pin_low(LED_PORT, LED_PIN);
        else
            md_gpio_set_pin_high(LED_PORT, LED_PIN);
    }
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


