/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    03 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          03 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes ------------------------------------------------------------------ */
#include <math.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

#define _PI acos(-1)

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint32_t data_freq = 2000U;
    uint32_t sampling_frequency = 16000U;
    uint32_t cut_off_frequency = 300U;
    uint32_t data_real = 0U;
    int index = 0U;

    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    __enable_irq();

    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_iir(RCU);

    uart_pin_init();
    uart_init();

    md_iir_coefficient_determines(IIR, sampling_frequency, cut_off_frequency, MD_IIR_2ST_ORDER_LOW_PASS_FILTER);

    while (1)
    {
        for (index = 0; index < sampling_frequency / data_freq; index++)
        {
            md_iir_data_calculate(&data_real, sin(2 * _PI * ((double)data_freq / (double)sampling_frequency)*index) + 0.5);
            md_iir_set_input_data(IIR, data_real);
            md_iir_set_start(IIR);

            while (md_iir_is_active_flag_busy(IIR));

            printf("INPUT DATA:%lf\t",sin(2 * _PI * ((double)data_freq / (double)sampling_frequency)*index) + 0.5);
            printf("IIR DONE DATA:%lf\r\n", md_iir_data_reverse_calculate(md_iir_get_output_data(IIR)));
        }
    }
}

int fputc(int c, FILE *f)
{
    while (!(UARTX->STAT & MD_UART_STATUS_TFEMPTY));

    UARTX->TXBUF = (uint8_t)c;

    return c;
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


