/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    09 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

#define SLAVE_ADDR  0xA0    /* Definition from the machine address */
#define MEM_ADDR    0x00    /* Define the machine from the initial memory address */

/* Private Variables --------------------------------------------------------- */

static char s_eeprom_txbuf[16] = "eeprom example!";     /* Length must be no more than one page (16 bytes) */
static char s_eeprom_rxbuf[16];

/* Public Variables ---------------------------------------------------------- */
md_i2c_init_t g_md_i2c;
uint8_t g_complete;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
* @brief  main.
* @param  None
* @retval None
* @note   When using, please pay attention to whether SCL, SDA and common ground are correctly connected
*/
int main(void)
{
    uint16_t i, cnt;

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    __enable_irq();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_i2c1(RCU);

    mcu_uart_init();    /*To initialize the UART module*/
    mcu_i2c_init();     /*Initialize the I2C module*/

    md_i2c_enable_master_write(I2C1);   /*Host write data model*/
    md_i2c_set_transmit_length(I2C1, strlen(s_eeprom_txbuf) + 1 + 1); /*The transmission of bytes*/
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR);  /*Set from the machine address*/
    md_i2c_set_start(I2C1);     /* + 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    printf("\r\nThe date written to eeprom address %d is -> %s\r\n", MEM_ADDR, s_eeprom_txbuf);
    md_i2c_set_tx_reg_data(I2C1, MEM_ADDR); /*Sent from the machine to be written to address*/
    cnt = 40000U;

    while (!md_i2c_is_active_txe(I2C1) && --cnt);  /*The transfer to complete*/

    for (i = 0; i < strlen(s_eeprom_txbuf) + 1; i++)
    {
        md_i2c_set_tx_reg_data(I2C1, s_eeprom_txbuf[i]);  /*To write data to send FIFO*/
        cnt = 40000U;

        while (!md_i2c_is_active_txe(I2C1) && --cnt);  /*The transfer to complete*/
    }

    cnt = 40000U;

    while (!md_i2c_is_active_tc(I2C1) && --cnt);  /*The transfer to complete*/

    md_i2c_set_stop(I2C1);  /*Send the stop bit*/

    for (i = 0U; i < 50000; i++); /* Inside block waiting for the eeprom write success */

    md_i2c_enable_master_write(I2C1);   /*Host write data model*/
    md_i2c_set_transmit_length(I2C1, 1);    /*The only transport address of the block 1 eeprom chip*/
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR);  /*Set from the machine address*/
    md_i2c_set_start(I2C1);     /*+ 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    md_i2c_set_tx_reg_data(I2C1, MEM_ADDR);

    cnt = 40000U;

    while (!md_i2c_is_active_tc(I2C1) && --cnt);  /*Waiting for completion of the inside address*/

    md_i2c_enable_master_read(I2C1);    /*Host read data model*/
    md_i2c_set_transmit_length(I2C1, strlen(s_eeprom_txbuf) + 1); /*The transmission of bytes*/
    md_i2c_set_start(I2C1);     /*restart*/

    for (i = 0U; i < strlen(s_eeprom_txbuf) + 1; i++)
    {
        cnt = 40000U;

        while (!md_i2c_is_active_rxne(I2C1) && --cnt);  /*Waiting to receive complete*/

        s_eeprom_rxbuf[i] = md_i2c_get_rx_reg_data(I2C1);
    }

    cnt = 40000U;

    while (!md_i2c_is_active_tc(I2C1) && --cnt);  /*The transfer to complete*/

    md_i2c_set_stop(I2C1);  /*Send the stop bit*/

    printf("The data read from eeprom address %d is  -> %s\r\n", MEM_ADDR, s_eeprom_rxbuf);

    if (!memcmp(s_eeprom_txbuf, s_eeprom_rxbuf, strlen(s_eeprom_txbuf) + 1))   /*Data is written and read*/
        printf("Read OK!\r\n");
    else
        printf("Read ERROR!\r\n");

    while (1);
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
