/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    09 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint8_t g_recv_buf[20];
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_send_buf[20];
uint8_t g_send_len = sizeof(g_send_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void init_i2c_pin(void)
{
    md_gpio_init_t md_gpio;
    md_gpio_init_struct(&md_gpio);

    /* Initialize scl pin */
    md_gpio.mode  = MD_GPIO_MODE_FUNCTION;
    md_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    md_gpio.pupd  = MD_GPIO_PUSH_UP;
    md_gpio.odrv  = MD_GPIO_DRIVING_8MA;
    md_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    md_gpio.type  = MD_GPIO_TYPE_CMOS;
    md_gpio.func  = MD_GPIO_FUNC_1;
    md_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &md_gpio);

    /* Initialize sda pin */
    md_gpio.mode  = MD_GPIO_MODE_FUNCTION;
    md_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    md_gpio.pupd  = MD_GPIO_PUSH_UP;
    md_gpio.odrv  = MD_GPIO_DRIVING_8MA;
    md_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    md_gpio.type  = MD_GPIO_TYPE_CMOS;
    md_gpio.func  = MD_GPIO_FUNC_1;
    md_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &md_gpio);

    return;
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(void)
{
    md_dma_config_t dma_config;

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    dma_config.src            = (void *)g_send_buf;
    dma_config.dst            = (void *)&I2C1->TXDATA;
    dma_config.size           = g_send_len;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msigsel        = MD_DMA_MSIGSEL_I2C1_TX;
    dma_config.dir            = MD_DMA_DIR_TO_PERH;
    md_dma_init(DMA, MD_DMA_CH_1, &dma_config);
}
/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(void)
{
    md_dma_config_t dma_config;

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    dma_config.src            = (void *)&I2C1->RXDATA;
    dma_config.dst            = (void *)g_recv_buf;
    dma_config.size           = g_recv_len;
    dma_config.src_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_BYTE;
    dma_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msigsel        = MD_DMA_MSIGSEL_I2C1_RX;
    dma_config.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA, MD_DMA_CH_0, &dma_config);
}

/**
* @brief  main.
* @param  None
* @retval None
* @note   When using, please pay attention to whether SCL, SDA and common ground are correctly connected
*/
int main(void)
{
    md_i2c_init_t md_i2c;

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    __enable_irq();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_i2c1(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_dmamux(RCU);

    init_i2c_pin();

    md_i2c_struct_init(&md_i2c);
    /* Initialize I2C */
    md_i2c.module       = MD_I2C_MODULE_SLAVE;
    md_i2c.addr_mode    = MD_I2C_ADDR_7BIT;
    md_i2c.clk_speed    = MD_I2C_FAST_MODE_MAX_CLK;
    md_i2c.dual_addr    = MD_I2C_DUALADDR_ENABLE;
    md_i2c.general_call = MD_I2C_GENERALCALL_DISABLE;
    md_i2c.no_stretch   = MD_I2C_NOSTRETCH_DISABLE;
    md_i2c.own_addr1    = OWN_ADDR << 1;
    md_i2c_init(I2C1, &md_i2c);

    /* receive */
    dma_recv_rx();

    /* send */
    dma_send_tx();

    while (1)
    {
        while(!md_i2c_is_active_flag_addr(I2C1));
        md_i2c_clear_flag_addr(I2C1);

        if (md_i2c_is_active_flag_read(I2C1))
        {
            md_i2c_enable_dma_transmit(I2C1);

            while(!md_dma_is_active_rif_tc(DMA, MD_DMA_CH_1));

            md_dma_clear_flag_tc(DMA, MD_DMA_CH_1);

            md_i2c_disable_dma_transmit(I2C1);
        }
        else
        {
            md_i2c_enable_dma_receive(I2C1);

            while(!md_dma_is_active_rif_tc(DMA, MD_DMA_CH_0));

            md_dma_clear_flag_tc(DMA, MD_DMA_CH_0);

            md_i2c_disable_dma_receive(I2C1);

            memcpy(g_send_buf, g_recv_buf, g_recv_len);
        }
    }
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
