/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    18 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          18 Apr. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
md_i2c_init_t g_md_i2c;
uint8_t g_recv_buf[20];
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_recv_inc = 0U;
uint8_t g_send_buf[20] = {0x55};
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_send_inc = 0U;
uint8_t g_complete = 1U;
uint8_t g_error_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void init_i2c_pin(void)
{
    md_gpio_init_t md_gpio;
    md_gpio_init_struct(&md_gpio);

    /* Initialize scl pin */
    md_gpio.mode  = MD_GPIO_MODE_FUNCTION;
    md_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    md_gpio.pupd  = MD_GPIO_PUSH_UP;
    md_gpio.odrv  = MD_GPIO_DRIVING_8MA;
    md_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    md_gpio.type  = MD_GPIO_TYPE_CMOS;
    md_gpio.func  = MD_GPIO_FUNC_1;
    md_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &md_gpio);

    /* Initialize sda pin */
    md_gpio.mode  = MD_GPIO_MODE_FUNCTION;
    md_gpio.odos  = MD_GPIO_OPEN_DRAIN;
    md_gpio.pupd  = MD_GPIO_PUSH_UP;
    md_gpio.odrv  = MD_GPIO_DRIVING_8MA;
    md_gpio.flt   = MD_GPIO_FILTER_DISABLE;
    md_gpio.type  = MD_GPIO_TYPE_CMOS;
    md_gpio.func  = MD_GPIO_FUNC_1;
    md_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &md_gpio);

    return;
}

/**
* @brief  main.
* @param  None
* @retval None
* @note   When using, please pay attention to whether SCL, SDA and common ground are correctly connected
*/
int main(void)
{
    uint16_t cnt;

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_mcu_irq_config(I2C1_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_i2c1(RCU);

    init_i2c_pin();

    md_i2c_struct_init(&g_md_i2c);
    /* Initialize I2C */
    g_md_i2c.module       = MD_I2C_MODULE_MASTER;
    g_md_i2c.addr_mode    = MD_I2C_ADDR_7BIT;
    g_md_i2c.clk_speed    = MD_I2C_FAST_MODE_MAX_CLK;
    g_md_i2c.dual_addr    = MD_I2C_DUALADDR_DISABLE;
    g_md_i2c.general_call = MD_I2C_GENERALCALL_DISABLE;
    g_md_i2c.no_stretch   = MD_I2C_NOSTRETCH_DISABLE;
    g_md_i2c.own_addr1    = 0x2D << 1;
    md_i2c_init(I2C1, &g_md_i2c);

    /* set slave address */
    md_i2c_set_slave_addr(I2C1, SLAVE_ADDR << 1);
    /* enable automatic end and automatically send stop bit */
    md_i2c_enable_auto_end(I2C1);

    /* Enable RXNE interrupt */
    md_i2c_enable_it_rxne(I2C1);
    /* Enable NACK interrupt */
    md_i2c_enable_it_nack(I2C1);

    while (1)
    {
        g_complete = 0;
        /* read */
        /* set transmission length */
        md_i2c_set_transmit_length(I2C1, g_recv_len);
        /* select master read mode */
        md_i2c_enable_master_read(I2C1);
        /* produce start signal */
        md_i2c_set_start(I2C1);

        cnt = 40000;
        while (!g_complete && (--cnt));

        cnt = 40000;
        while (md_i2c_is_active_busy(I2C1) && (--cnt));

        memcpy(g_send_buf, g_recv_buf, g_recv_len);

        g_complete = 0;
        /* write */
        g_error_flag = 0;
        /* set transmission length */
        md_i2c_set_transmit_length(I2C1, g_send_len);
        /* select master write mode */
        md_i2c_enable_master_write(I2C1);
        /* produce start signal */
        md_i2c_set_start(I2C1);

        /* Enable TXE interrupt */
        md_i2c_enable_it_txe(I2C1);

        cnt = 40000;
        while (!g_complete && (--cnt));

        /* Disable TXE interrupt */
        md_i2c_disable_it_txe(I2C1);

        cnt = 40000;
        while (md_i2c_is_active_busy(I2C1) && (--cnt));

        md_delay_1ms(2000);
    }
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
