/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    25 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_config(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize adc gpio pin */
    gpio_init.mode  = MD_GPIO_MODE_ANALOG;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(ADC12_CHANNEL1_GPIO_PORT, ADC12_CHANNEL1_PIN, &gpio_init);

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.odrv = MD_GPIO_DRIVING_8MA;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_5;
    md_gpio_init(CH1_PORT, CH1_PIN, &gpio_init);
    md_gpio_init(CH1N_PORT, CH1N_PIN, &gpio_init);

    return;
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_config(void)
{
    md_adc_init_typedef ADC_Init;
    md_adc_input_channel_config_typedef config;
    md_adc_regular_trigger_config_typedef trigger_config;
    memset(&ADC_Init, 0x0, sizeof(md_adc_init_typedef));
    memset(&config, 0x0, sizeof(md_adc_input_channel_config_typedef));
    memset(&trigger_config, 0x0, sizeof(md_adc_regular_trigger_config_typedef));

    ADC_Init.clock_source          = MD_ADC_KERNEL_AHBCLK;
    ADC_Init.clock_source_prescale = MD_ADC_CCR_PRESCALE_64;
    ADC_Init.adc_lowpower_mode     = MD_ADC_LOWPOWER_DISABLE;
    ADC_Init.data_resolution       = MD_ADC_CFG1_RESOL_12BIT;
    ADC_Init.data_align            = MD_ADC_CFG1_ALIGN_RIGHT;
    md_adc_init(ADC1, ADC12_COMM, &ADC_Init);

    config.sample_time_trigger_mode_en = MD_ADC_SAMPLE_TIME_TRIGGER_MODE_DISABLE;
    config.bulb_mod_en = MD_ADC_BULB_MODE_DISABLE;
    config.extra_increase_sample_time_en = MD_ADC_EXTRA_INCREASE_SAMPLE_TIME_DISABLE;
    config.regular_continute_mode = MD_ADC_CFG1_REGULAR_SINGLE;

    config.regular_channel_array.array_1 = MD_ADC_CH1;
    config.regular_channel_length = MD_ADC_RSQLEN_1;

    config.sample_time_config.ch1 = 0xFF;
    md_adc_input_channel_config(ADC1, &config);

    trigger_config.regular_trigger_mode = MD_ADC_CFG1_TRIG_EXT_RISING;
    trigger_config.regular_external_trigger_source = MD_ADC_CFG1_REXT_AD16C6T1_CH1;
    md_adc_regular_trigger_config(ADC1, &trigger_config);

    md_adc_enable_it_regular_channel_conversion_end(ADC1);
}

/**
  * @brief:  Initialize AD16C4T to
  *          6 PWM/dead time/break funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_oc_init_t oc_init;
    md_timer_break_dead_time_t brake_setting;

    md_timer_base_struct_init(&ad16c4t_init);        /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler = 72 - 1;                 /* clk_count: 72 */
    ad16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1;      /* working clock of dead time and filter */
    ad16c4t_init.mode = MD_TIMER_CNT_MODE_CENTER1;   /* count up */
    ad16c4t_init.period = 50000 - 1;                  /* period is 5000 count clock */
    ad16c4t_init.re_cnt = 1 - 1;                     /* 0 repeat count */
    md_timer_base_set_config(AD16C6T1, &ad16c4t_init);

    md_timer_oc_struct_init(&oc_init);
    oc_init.ocstate = MD_TIMER_OCSTATE_ENABLE;    /* enable CHxO compare output */
    oc_init.ocnstate = MD_TIMER_OCSTATE_ENABLE;   /* enable CHxON compare output */
    oc_init.oc_mode = MD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    oc_init.oc_polarity = MD_TIMER_OC_POLARITY_HIGH;    /* CHxO compare output polarity is high */
    oc_init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;  /* CHxON compare output polarity is high */
    oc_init.oc_idle = MD_TIMER_OC_IDLE_RESET;           /* CHxO idle is low */
    oc_init.ocn_idle = MD_TIMER_OCN_IDLE_SET;         /* CHxON idle is low */
    oc_init.pulse = 25000 - 1;    /* CH1 compare value is 1500 - 1 */
    md_timer_oc_init(AD16C6T1, MD_TIMER_CHANNEL_1, &oc_init);

    brake_setting.off_run = ENABLE;        /* Enable off state in run mode */
    brake_setting.off_idle = ENABLE;       /* Enable off state in idle mode */
    brake_setting.lock_level = MD_TIMER_LOCK_LEVEL_OFF;  /* Lock level off */
    brake_setting.dead_time = 0xF0;        /* Dead time, [0x0, 0xFF] */
    brake_setting.break_state = DISABLE;    /* Break state */
    brake_setting.polarity = MD_TIMER_BREAK_POLARITY_LOW;    /* Break input polarity */
    brake_setting.auto_out = ENABLE;       /* Enable automatic output */
    md_timer_bdtr_init(AD16C6T1, &brake_setting);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    md_mcu_irq_config(ADC_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_ad16c6t1(RCU);
    md_rcu_enable_adc(RCU);

    gpio_pin_config();
    init_timer();
    adc_config();

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    md_adc_set_start_regular(ADC1);
    md_timer_enable_counter_cnten(AD16C6T1);     /* enable AD16C6T1 */

    while (1);
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
