/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    25 July 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          25 July 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uint16_t g_adc_result_temp[4] = {0U};
uint16_t g_adc_result[4] = {0U};
uint8_t g_dma_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void gpio_pin_config(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize adc gpio pin */
    gpio_init.mode  = MD_GPIO_MODE_ANALOG;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(ADC12_CHANNEL0_GPIO_PORT, ADC12_CHANNEL0_PIN, &gpio_init);
    md_gpio_init(ADC12_CHANNEL1_GPIO_PORT, ADC12_CHANNEL1_PIN, &gpio_init);
    md_gpio_init(ADC12_CHANNEL2_GPIO_PORT, ADC12_CHANNEL2_PIN, &gpio_init);
    md_gpio_init(ADC12_CHANNEL3_GPIO_PORT, ADC12_CHANNEL3_PIN, &gpio_init);

    return;
}

/**
  * @brief  Configure the ADC.
  * @param  None
  * @retval None
  */
void adc_config(void)
{
    md_adc_init_typedef ADC_Init;
    md_adc_input_channel_config_typedef config;
    md_adc_regular_trigger_config_typedef trigger_config;
    memset(&ADC_Init, 0x0, sizeof(md_adc_init_typedef));
    memset(&config, 0x0, sizeof(md_adc_input_channel_config_typedef));
    memset(&trigger_config, 0x0, sizeof(md_adc_regular_trigger_config_typedef));

    ADC_Init.clock_source          = MD_ADC_KERNEL_AHBCLK;
    ADC_Init.clock_source_prescale = MD_ADC_CCR_PRESCALE_64;
    ADC_Init.adc_lowpower_mode     = MD_ADC_LOWPOWER_DISABLE;
    ADC_Init.data_resolution       = MD_ADC_CFG1_RESOL_12BIT;
    ADC_Init.data_align            = MD_ADC_CFG1_ALIGN_RIGHT;
    md_adc_init(ADC1, ADC12_COMM, &ADC_Init);

    md_adc_enable_dma_access(ADC1);

    config.sample_time_trigger_mode_en = MD_ADC_SAMPLE_TIME_TRIGGER_MODE_DISABLE;
    config.bulb_mod_en = MD_ADC_BULB_MODE_DISABLE;
    config.extra_increase_sample_time_en = MD_ADC_EXTRA_INCREASE_SAMPLE_TIME_DISABLE;
    config.regular_continute_mode = MD_ADC_CFG1_REGULAR_CONTINUOUS;

    config.regular_channel_length = MD_ADC_RSQLEN_4;
    config.regular_channel_array.array_1 = MD_ADC_CH0;
    config.regular_channel_array.array_2 = MD_ADC_CH1;
    config.regular_channel_array.array_3 = MD_ADC_CH2;
    config.regular_channel_array.array_4 = MD_ADC_CH3;

    config.sample_time_config.ch0 = 0xFF;
    config.sample_time_config.ch1 = 0xFF;
    config.sample_time_config.ch2 = 0xFF;
    config.sample_time_config.ch3 = 0xFF;
    md_adc_input_channel_config(ADC1, &config);

    trigger_config.regular_trigger_mode = MD_ADC_CFG1_TRIG_SOFTWARE;
    md_adc_regular_trigger_config(ADC1, &trigger_config);
}

/**
  * @brief  Configure the DMA.
  * @param  None
  * @retval None
  */
void dma_init(void)
{
    md_dma_config_t dma_config;

    memset(&dma_config, 0x0, sizeof(md_dma_config_t));
    dma_config.src            = (void *)(&ADC1->DR);
    dma_config.dst            = (void *)g_adc_result_temp;
    dma_config.size           = sizeof(g_adc_result_temp);
    dma_config.src_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.dst_data_width = MD_DMA_DATA_SIZE_HALFWORD;
    dma_config.src_inc        = MD_DMA_DATA_INC_DISABLE;
    dma_config.dst_inc        = MD_DMA_DATA_INC_ENABLE;
    dma_config.R_power        = MD_DMA_R_POWER_1;
    dma_config.priority       = MD_DMA_LOW_PRIORITY;
    dma_config.mem_to_mem     = DISABLE;
    dma_config.circle_mode    = ENABLE;
    dma_config.msigsel        = MD_DMA_MSIGSEL_ADC1;
    dma_config.dir            = MD_DMA_DIR_TO_SRAM;
    md_dma_init(DMA, MD_DMA_CH_0, &dma_config);

    md_dma_enable_it_tc(DMA, MD_DMA_CH_0);
}

/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    uint8_t i;

    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    md_mcu_irq_config(DMA1_CH0_IRQn, 3, ENABLE);
    __enable_irq();

    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_dma1(RCU);
    md_rcu_enable_dmamux(RCU);
    md_rcu_enable_adc(RCU);

    dma_init();
    gpio_pin_config();
    adc_config();

    /* Initialize SysTick Interrupt */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    /* Enable ADC group REG conversion trigger source internal (SW start)*/
    md_adc_set_start_regular(ADC1);

    while (1)
    {
        if (g_dma_flag)
        {
            g_dma_flag = 0;

            /* calculate value */
            for (i = 0; i < sizeof(g_adc_result); i++)
                g_adc_result[i] = g_adc_result_temp[i] * VDD / 4095;
        }
    }
}

/**
  * @} Projects_Examples_MD
  */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
