/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    09 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Apr. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/

#include "main.h"
#include "printf.h"

/* Private types --------------------------------------------------------------*/
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin */
    gpio_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART */
    uart_init.baud        = 115200;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UARTX, &uart_init);
}

volatile uint32_t g_systick=0;

__STATIC_INLINE uint32_t SYSTICK_IsActiveCounterFlag(void)
{
    return ((SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk) == (SysTick_CTRL_COUNTFLAG_Msk));
}
uint64_t getCurrentMicros(void)
{
    /* Ensure COUNTFLAG is reset by reading SysTick control and status register */ 
    /* When using this function for serial connection, you need to configure the system tick frequency to 10kHz(100us).*/
    uint32_t m; 
    uint32_t tms; 
    __IO uint32_t u;
    
    SYSTICK_IsActiveCounterFlag();
    m= g_systick;
    tms= SysTick->LOAD + 1;
    u= tms - SysTick->VAL;
    if (SYSTICK_IsActiveCounterFlag())
    {
        m = g_systick;
        u = tms - SysTick->VAL;
    }
    return ((uint64_t)m)*100000 + (u * 1000) / tms;

}

extern int benchmark_coremark_main(void);
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
    __disable_irq();

    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_24M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 24000000U);
    md_fc_set_read_latency(FC, MD_FC_WAIT_LESS_THAN_24MHz);

    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);

    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    SysTick_Config(SystemFrequency_SysClk / 10U);
    NVIC_SetPriority(SysTick_IRQn, 3);
    SysTick->VAL  = 0;

    __enable_irq();

    uart_pin_init();
    uart_init();

    printf("Startup\n");

    benchmark_coremark_main();

    while (1);
}

void _putchar(char chr)
{
    if (chr=='\n')
        _putchar('\r');

    while (!(UARTX->STAT & MD_UART_STATUS_TFEMPTY));
    UARTX->TXBUF = (uint8_t)chr;
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


