/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Oct 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Oct 2024     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"

/* Public Variables ---------------------------------------------------------- */
uint8_t g_uart_rx_buf[64];
uint8_t g_uart_rx_cnt = 0;

uint8_t g_go_flag = 0;
uint16_t g_frame_tick = 0;
/* Public Function ---------------------------------------------------------- */
typedef  void (*FunVoidType)(void);
void uart_init(void);
void led_gpio_init(void);

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  uart recv callback.
  * @param  None
  * @retval None
  */
void uart_recv_callback(void)
{
    g_uart_rx_buf[g_uart_rx_cnt++] = md_uart_recv_data8(UART1);
	
    set_frame_tick(FRAME_INTERVAL);
}

/**
  * @brief  frame tick callback, for detect frame timeout.
  * @param  None
  * @retval None
  */
void frame_tick_callback(void)
{
    if (g_frame_tick != 0)
    {
        g_frame_tick--;

        if (g_frame_tick == 0)
        {
            g_uart_rx_cnt = 0;
        }
    }
}

/**
  * @brief  set frame timeout tick.
  * @param  value: timeout tick value
  * @retval None
  */
void set_frame_tick(uint16_t value)
{
    g_frame_tick = value;
}

/**
  * @brief  get frame timeout tick.
  * @param  None
  * @retval time remaining after timeout
  */
uint16_t get_frame_tick(void)
{
    return g_frame_tick;
}

/**
  * @brief  define the peripheral register clear function.
  * @param  None
  * @retval None
  */
static void sfr_reset(void)
{
    md_rcu_enable_gpioa_reset(RCU);
    md_rcu_enable_gpiob_reset(RCU);
    md_rcu_enable_gpioc_reset(RCU);
    md_rcu_enable_uart1_reset(RCU);
    md_rcu_enable_uart2_reset(RCU);

    md_rcu_disable_gpioa_reset(RCU);
    md_rcu_disable_gpiob_reset(RCU);
    md_rcu_disable_gpioc_reset(RCU);
    md_rcu_disable_uart1_reset(RCU);
    md_rcu_disable_uart2_reset(RCU);
}

/**
  * @brief  define the function used to jump to app program.
  * @param  None
  * @retval None
  */
static void fsm_go(uint32_t para)
{
    uint32_t addr;
	
    /* Lock FC_CMD Register */
    md_fc_lock();

    if(para == GO_APP)
    {
        addr = APP_ADDR;
    } 
    else if(para == GO_BOOT) 
    {
        addr = BOOT_ADDR;
    }    

		sfr_reset();
		
    /* REMAP */
    /* 0x00: EFLASH         -> Remap EFLASH Any 4K */
    /* 0x01: Bootloader     -> Remap EFLASH 0x00000000 */
    /* 0x02: SRAM           -> Remap SRAM */

    /* Remap EFLASH */
    md_syscfg_set_memory_mapping(SYSCFG, MD_SYSCFG_MEMMOD_MAIN);
    md_syscfg_set_flash_remap_base(SYSCFG, addr / (4 * 2 * MD_FC_PC_EF_MPAGESZ));
    md_syscfg_enable_memory_remap(SYSCFG);
		
    /* Reset MCU */
    NVIC_SystemReset();

    while (1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock: use default internal HRC */
    md_rcu_pll_config(MD_RCU_PLLSRC_HRC, MD_RCU_PLLCLK_72M);
    md_rcu_clock_config(MD_RCU_SW_SYSCLK_PLL, 72000000U);

    /* Initialize Systick to tick per 1ms */
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
    md_init_1ms_tick();

    __enable_irq();
    md_mcu_irq_config(UART1_IRQn, 3, ENABLE);

    md_rcu_enable_gpioa(RCU);
    md_rcu_enable_gpiob(RCU);
    md_rcu_enable_gpioc(RCU);
    md_rcu_enable_uart1(RCU);
    md_rcu_enable_uart2(RCU);

    /* Init communication parameters */
    uart_init();
		led_gpio_init();
	
    printf("\r\nES32M0502 application start\r\n");
    printf("SystemFrequency_SysClk:%d\r\n", SystemFrequency_SysClk);
    printf("SystemFrequency_AHBClk:%d\r\n", SystemFrequency_AHBClk);
    printf("SystemFrequency_APBClk:%d\r\n", SystemFrequency_APBClk);

    while (1)
    {
        if (g_uart_rx_cnt >= 8)
        {
            if (g_uart_rx_buf[0] == 0x3F && g_uart_rx_buf[1] == 0xA1 && g_uart_rx_buf[2] == 0x5E &&
								g_uart_rx_buf[3] == 0xFF && g_uart_rx_buf[4] == 0xFF && g_uart_rx_buf[5] == 0xFF &&
								g_uart_rx_buf[6] == 0xFD && g_uart_rx_buf[7] == 0xFA)
            {
                md_uart_set_send_data8(UART1, POS_ACK);

                while (md_uart_is_active_flag_tsbusy(UART1));

                /* wait for uart send data */
                md_delay_1ms(50);

                fsm_go(GO_BOOT);
            }
						else
						{
								g_uart_rx_cnt = 0;
						}
        }
    }
}

/**
  * @brief uart init
  * @param None
  * @retval None
  */
void uart_init(void)
{
    md_gpio_init_t gpiox_init;
    md_uart_init_t uartx_init;

    md_gpio_init_struct(&gpiox_init);

    /* Initialize uart1 tx pin */
    gpiox_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpiox_init.odos  = MD_GPIO_PUSH_PULL;
    gpiox_init.pupd  = MD_GPIO_FLOATING;
    gpiox_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpiox_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpiox_init.type  = MD_GPIO_TYPE_CMOS;
    gpiox_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpiox_init);

    /* Initialize uart1 rx pin */
    gpiox_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpiox_init.odos  = MD_GPIO_PUSH_PULL;
    gpiox_init.pupd  = MD_GPIO_PUSH_UP;
    gpiox_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpiox_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpiox_init.type  = MD_GPIO_TYPE_CMOS;
    gpiox_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART_RX_PORT, UART_RX_PIN, &gpiox_init);

    /* Initialize uart2 tx pin */
    gpiox_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpiox_init.odos  = MD_GPIO_PUSH_PULL;
    gpiox_init.pupd  = MD_GPIO_FLOATING;
    gpiox_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpiox_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpiox_init.type  = MD_GPIO_TYPE_CMOS;
    gpiox_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(DEBUG_UART_TX_PORT, DEBUG_UART_TX_PIN, &gpiox_init);

    /* Initialize uart2 rx pin */
    gpiox_init.mode  = MD_GPIO_MODE_FUNCTION;
    gpiox_init.odos  = MD_GPIO_PUSH_PULL;
    gpiox_init.pupd  = MD_GPIO_PUSH_UP;
    gpiox_init.odrv  = MD_GPIO_DRIVING_8MA;
    gpiox_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpiox_init.type  = MD_GPIO_TYPE_CMOS;
    gpiox_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(DEBUG_UART_RX_PORT, DEBUG_UART_RX_PIN, &gpiox_init);

    md_uart_init_struct(&uartx_init);

    /* Initialize UART */
    uartx_init.baud        = 115200;
    uartx_init.word_length = MD_UART_WORD_LENGTH_8B;
    uartx_init.stop_bits   = MD_UART_STOP_BITS_1;
    uartx_init.parity      = MD_UART_PARITY_NONE;
    uartx_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uartx_init.mode        = MD_UART_MODE;
    md_uart_init(UART1, &uartx_init);
    md_uart_init(UART2, &uartx_init);

    md_uart_clear_flag_rfnempty(UART1);

    md_uart_enable_it_rfnempty(UART1);
}

/**
  * @brief led init
  * @param None
  * @retval None
  */
void led_gpio_init(void)
{
    md_gpio_init_t led_init;

    /* Initialize led 1 led 2 led 3 led 4 */
    led_init.mode  = MD_GPIO_MODE_OUTPUT;
    led_init.odos  = MD_GPIO_PUSH_PULL;
    led_init.pupd  = MD_GPIO_FLOATING;
    led_init.odrv  = MD_GPIO_DRIVING_8MA;
    led_init.flt   = MD_GPIO_FILTER_DISABLE;
    led_init.type  = MD_GPIO_TYPE_CMOS;
    led_init.func  = MD_GPIO_FUNC_0;
    md_gpio_init(LED_1_PORT, LED_1_PIN, &led_init);
    md_gpio_init(LED_2_PORT, LED_2_PIN, &led_init);
    md_gpio_init(LED_3_PORT, LED_3_PIN, &led_init);
    md_gpio_init(LED_4_PORT, LED_4_PIN, &led_init);
	
		md_gpio_write_pin(LED_1_PORT, LED_1_PIN, 1);
		md_gpio_write_pin(LED_2_PORT, LED_2_PIN, 1);
		md_gpio_write_pin(LED_3_PORT, LED_3_PIN, 0);
		md_gpio_write_pin(LED_4_PORT, LED_4_PIN, 0);
}

/**
  * @brief  Retargets the C library printf function to the UART2.
  * @param  None
  * @retval None
  */
int fputc(int ch, FILE *f)
{
    /* waiting until sending over */
    while (md_uart_is_active_flag_tffull(UART2));

		md_uart_clear_it_tbc(UART2);
    md_uart_set_send_data8(UART2, (uint8_t)ch);
	
		while (!md_uart_is_active_it_tbc(UART2));

    return ch;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
