/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */


#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

usart_handle_t h_usart;
uint8_t frame_flag = 0;
uint8_t frame_buf[BUFFER_SIZE];
uint8_t frame_len = 0;

/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
void usart_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_3;
    ald_gpio_init(GPIOA, GPIO_PIN_2, &x);

    /* Initialize rx pin */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_3;
    ald_gpio_init(GPIOA, GPIO_PIN_3, &x);

    return;
}

/**
  * @brief  dma init.
  * @param hperh: USART instance pointer.
  * @retval None.
  */
void dma_init(usart_handle_t *hperh)
{
    hperh->hdmarx.perh = DMA0;

    /* Configure callback function */
    hperh->hdmarx.cplt_cbk = NULL;
    hperh->hdmarx.cplt_arg = NULL;
    hperh->hdmarx.err_cbk  = NULL;
    hperh->hdmarx.err_arg  = NULL;

    /* Configure DMA Receive */
    ald_dma_config_struct(&hperh->hdmarx.config);
    hperh->hdmarx.config.src     = (void *)&hperh->perh->DATA;
    hperh->hdmarx.config.dst     = (void *)&frame_buf[1];
    hperh->hdmarx.config.size    = BUFFER_SIZE - 1;
    hperh->hdmarx.config.src_inc = DMA_DATA_INC_NONE;
    hperh->hdmarx.config.dst_inc = DMA_DATA_INC_BYTE;
    hperh->hdmarx.config.msel    = hperh->perh == USART0 ? DMA_MSEL_USART0 : DMA_MSEL_USART1;
    hperh->hdmarx.config.msigsel = DMA_MSIGSEL_USART_RNR;
    hperh->hdmarx.config.channel = 0;

    ald_dma_config_basic(&hperh->hdmarx);
}

/**
  * @brief  usart receive a frame by dma.
  * @retval None.
  */
void usart_rec_frame_by_dma(void)
{
    ald_usart_interrupt_config(&h_usart, USART_IT_RXNE, ENABLE);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_mcu_irq_config(USART0_IRQn, 0, ENABLE);

    /* clear usart_handle_t structure */
    memset(&h_usart, 0x0, sizeof(h_usart));
    /* Initialize pin */
    usart_pin_init();

    /* Initialize USART */
    h_usart.perh             = USART0;
    h_usart.init.baud        = 115200;
    h_usart.init.word_length = USART_WORD_LENGTH_8B;
    h_usart.init.stop_bits   = USART_STOP_BITS_1;
    h_usart.init.parity      = USART_PARITY_NONE;
    h_usart.init.mode        = USART_MODE_TX_RX;
    h_usart.init.fctl        = USART_HW_FLOW_CTL_NONE;
    h_usart.tx_cplt_cbk      = NULL;
    h_usart.rx_cplt_cbk      = NULL;
    h_usart.tx_rx_cplt_cbk   = NULL;
    h_usart.error_cbk        = NULL;
    ald_usart_init(&h_usart);
    dma_init(&h_usart);
    usart_rec_frame_by_dma();

    while (1)
    {
        if (frame_flag == 1)
        {
            frame_flag = 0;
            ald_usart_send_by_dma(&h_usart, frame_buf, frame_len, 1);
            usart_rec_frame_by_dma();
        }
    }
}



/**
  * @}
  */
/**
  * @}
  */
