/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

usart_handle_t  h_usart;

uint8_t tx_buf[8];
uint8_t rx_buf[8];
uint8_t tx_flag;
uint8_t rx_flag;
uint32_t _cnt;

/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
void usart_pin_init(void)
{
    gpio_init_t x;

    /* Initialize tx pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_3;
    ald_gpio_init(GPIOA, GPIO_PIN_2, &x);

    /* Initialize rx pin */
    x.mode = GPIO_MODE_INPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_0_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_3;
    ald_gpio_init(GPIOA, GPIO_PIN_3, &x);

    return;
}
/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void usart_send_complete(usart_handle_t *arg)
{
    tx_flag = 1;
    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void usart_recv_complete(usart_handle_t *arg)
{
    rx_flag = 1;
    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void usart_error(usart_handle_t *arg)
{
    tx_flag = 0;
    rx_flag = 0;
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* clear usart_handle_t structure */
    memset(&h_usart, 0x0, sizeof(h_usart));

    /* Initialize pin */
    usart_pin_init();

    /* Initialize tx_buf */
    memset(tx_buf, 0x00, sizeof(tx_buf));

    /* Initialize usart */
    h_usart.perh             = USART0;
    h_usart.init.baud        = 115200;
    h_usart.init.word_length = USART_WORD_LENGTH_8B;
    h_usart.init.stop_bits   = USART_STOP_BITS_1;
    h_usart.init.parity      = USART_PARITY_NONE;
    h_usart.init.mode        = USART_MODE_TX_RX;
    h_usart.init.fctl        = USART_HW_FLOW_CTL_NONE;
    h_usart.tx_cplt_cbk      = usart_send_complete;
    h_usart.rx_cplt_cbk      = usart_recv_complete;
    h_usart.error_cbk        = usart_error;
    ald_usart_init(&h_usart);

    /* Enable USART0 interrupt */
    ald_mcu_irq_config(USART0_IRQn, 3, ENABLE);

    tx_flag = 1;
    rx_flag = 1;

    while (1)
    {
        if (tx_flag)
        {
            tx_flag = 0;
            memset(tx_buf, (uint8_t)(_cnt & 0xFF), sizeof(tx_buf));
            ald_usart_send_by_dma(&h_usart, tx_buf, 8, 0);
        }

        if (rx_flag)
        {
            rx_flag = 0;
            ald_usart_recv_by_dma(&h_usart, rx_buf, 8, 1);
        }

        ++_cnt;
        ald_delay_ms(100);
    }
}

/**
  * @}
  */
/**
  * @}
  */
