/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0.0
  * @date    11 Nov 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          11 Nov 2023     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint32_t nm_value;
adc_handle_t h_adc;

/**
  * @brief  Adc normal convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void normal_complete_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc insert convert complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void insert_complete_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc level out of window complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void wdg_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc error complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void error_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Adc over complete.
  * @param  arg: Pointer to adc_handle_t structure.
  * @retval None.
  */
static void ovr_cbk(adc_handle_t *arg)
{
    return;
}

/**
  * @brief  Initializate ADC.
  * @retval None.
  */
void adc_init(void)
{
    adc_nch_conf_t nm_config;

    /* clear structures */
    memset(&h_adc, 0x0, sizeof(adc_handle_t));
    memset(&nm_config, 0x0, sizeof(adc_nch_conf_t));

    /* Initialize adc */
    h_adc.perh            = ADC0;
    h_adc.init.align      = ADC_DATAALIGN_RIGHT;
    h_adc.init.scan       = DISABLE;
    h_adc.init.cont       = DISABLE;
    h_adc.init.nch_nr     = ADC_NCH_NR_1;
    h_adc.init.disc       = ADC_ALL_DISABLE;
    h_adc.init.disc_nr    = ADC_DISC_NR_1;
    h_adc.init.data_bit   = ADC_CONV_BIT_12;
    h_adc.init.div        = ADC_CKDIV_32;
    h_adc.init.nche_sel   = ADC_NCHESEL_MODE_ONE;
    h_adc.init.n_ref      = ADC_NEG_REF_VSS;
    h_adc.init.p_ref      = ADC_POS_REF_2V;
    h_adc.normal_cplt_cbk = normal_complete_cbk;
    h_adc.insert_cplt_cbk = insert_complete_cbk;
    h_adc.wdg_cbk         = wdg_cbk;
    h_adc.error_cbk       = error_cbk;
    h_adc.ovr_cbk         = ovr_cbk;
    ald_adc_init(&h_adc);

    /* Initialize normal convert channel */
    nm_config.ch   = ADC_CHANNEL_16;
    nm_config.idx  = ADC_NCH_IDX_1;
    nm_config.samp = ADC_SAMPLETIME_4;
    ald_adc_normal_channel_config(&h_adc, &nm_config);

    return;
}

/**
  * @brief  Get the average value of the normal channel.
  * @retval Average voltage.
  */
uint32_t get_normal_average_voltage(void)
{
    uint32_t i, tmp, min = 0xFFFF, max = 0, vol, sum = 0;

    for (i = 0; i < 18; ++i)
    {
        /* Start normal convert */
        ald_adc_normal_start(&h_adc);

        /* Polling convert finish and read result */
        if (ald_adc_normal_poll_for_conversion(&h_adc, 5000) == OK)
            tmp = ald_adc_normal_get_value(&h_adc);
        else
            return 0;

        max  = tmp > max ? tmp : max;
        min  = tmp < min ? tmp : min;
        sum += tmp;
    }

    sum = sum - max - min;
    vol = sum >> 4;

    return vol;
}

/**
 * @brief HRC parameters Reset
 * @parm  Temp: Current Temperature
 * @retval None
 */
void SetHRC_5V(float Temperature)
{
    uint32_t x1, x2, y1, y2;
    int32_t x, y;
    int32_t tmp;
    float K, B, X, tmp1;
    uint32_t code;
    uint16_t code_step;

    code_step = 65;//K code *1024 (ns)
//  x1 = ADCTemp_Standard_30;
//  x2 = ADCTemp_Standard_85;
    x1 = 25;
    x2 = 85;
    y1 = HRC_Standard_30 ;
    y2 = HRC_Standard_85 ;
    code = HRC_Standard & 0X3FF;//Default value

    y = y1 - y2;
    x = x1 - x2;
    K = (float)y / (float)x;

    X = Temperature;
    B = (X - 25) * K;
    tmp1 = B / (float)code_step;


    if (tmp1 > 0)
    {
        if (((tmp1 + 0.5) > (int)tmp1) == TRUE)
        {
            tmp = code + (int)tmp1 + 1;
        }
        else
        {
            tmp = code + (int)tmp1;
        }
    }
    else if (tmp1 == 0)
    {
        tmp = code;
    }
    else
    {
        if (((tmp1 - 0.5) < (int)tmp1) == TRUE)
        {
            tmp = code + (int)tmp1 - 1;
        }
        else
        {
            tmp = code + (int)tmp1;
        }
    }

    if (((tmp < 1024) == TRUE) & ((tmp > 0) == TRUE))
    {
        code = tmp;
        code |= HRC_Standard & 0XFC00;
        code |= 0x55AA << 16 ;
        // unlock
        *((volatile uint32_t *)(0x40080000)) = 0x55AA6996;
        *((volatile uint32_t *)(0x40080100)) = 0x5A962814;
        *((volatile uint32_t *)(0x40080100)) = 0xE7CB69A5;
        *((volatile uint32_t *)(Reg_Address)) = code;
    }
    else if ((tmp > 1024) == TRUE)
    {
        code = 0x3FF;
        code |= HRC_Standard & 0XFC00;
        code |= 0x55AA << 16 ;
        // unlock
        *((volatile uint32_t *)(0x40080000)) = 0x55AA6996;
        *((volatile uint32_t *)(0x40080100)) = 0x5A962814;
        *((volatile uint32_t *)(0x40080100)) = 0xE7CB69A5;
        *((volatile uint32_t *)(Reg_Address)) = code;
    }
    else if ((tmp < 0) == TRUE)
    {
        code = 0;
        code |= HRC_Standard & 0XFC00;
        code |= 0x55AA << 16 ;
        // unlock
        *((volatile uint32_t *)(0x40080000)) = 0x55AA6996;
        *((volatile uint32_t *)(0x40080100)) = 0x5A962814;
        *((volatile uint32_t *)(0x40080100)) = 0xE7CB69A5;
        *((volatile uint32_t *)(Reg_Address)) = code;
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    float Temperature;

    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

    /* Enable peripheral clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Initialize ADC */
    adc_init();

    nm_value = get_normal_average_voltage();

    Temperature = ald_adc_tsensor_get_temperature(nm_value);

    SetHRC_5V(Temperature);

    ADC_DISABLE(&h_adc);

    while (1)
    {
        ald_delay_ms(1000);
    }
}

/**
  * @}
  */

/**
  * @}
  */
