/**
  *********************************************************************************
  *
  * @file    tk_low_power.c
  * @brief   TK low power Process
  *
  * @version V1.0
  * @date    06 Sep 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          06 Sep 2023     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  *********************************************************************************
  */
#include <string.h>
#include "ald_cmu.h"
#include "ald_wdt.h"
#include "tk_low_power.h"
#include "tk_lib.h"
#include "tk_config.h"
#include "tk_uart.h"

#if TK_LOWPOWER_MODE == ON

/* 0x0: Enter sleep
 * 0x1: TK key active, no sleep
 */
uint8_t __sleep_flag;
uint8_t __lp_map[TK_NUM_LP];
tk_lp_arr_t lp_arr[TK_NUM_LP];
uint32_t _lp_cnt = 0;
uint8_t __lp_base_cnt;

/* Channel table for use as tk channel */
const uint8_t ch_table_lp[TK_NUM_LP] =
{
#if TK_NUM_LP >0
    TK_CHANNEL_0_LP,
#endif
#if TK_NUM_LP >1
    TK_CHANNEL_1_LP,
#endif
#if TK_NUM_LP >2
    TK_CHANNEL_2_LP,
#endif
#if TK_NUM_LP >3
    TK_CHANNEL_3_LP,
#endif
#if TK_NUM_LP >4
    TK_CHANNEL_4_LP,
#endif
#if TK_NUM_LP >5
    TK_CHANNEL_5_LP,
#endif
#if TK_NUM_LP >6
    TK_CHANNEL_6_LP,
#endif
#if TK_NUM_LP >7
    TK_CHANNEL_7_LP,
#endif
#if TK_NUM_LP >8
    TK_CHANNEL_8_LP,
#endif
#if TK_NUM_LP >9
    TK_CHANNEL_9_LP,
#endif
#if TK_NUM_LP >10
    TK_CHANNEL_10_LP,
#endif
#if TK_NUM_LP >11
    TK_CHANNEL_11_LP,
#endif
#if TK_NUM_LP >12
    TK_CHANNEL_12_LP,
#endif
#if TK_NUM_LP >13
    TK_CHANNEL_13_LP,
#endif
#if TK_NUM_LP >14
    TK_CHANNEL_14_LP,
#endif
#if TK_NUM_LP >15
    TK_CHANNEL_15_LP,
#endif
#if TK_NUM_LP >16
    TK_CHANNEL_16_LP,
#endif
#if TK_NUM_LP >17
    TK_CHANNEL_17_LP,
#endif
#if TK_NUM_LP >18
    TK_CHANNEL_18_LP,
#endif
#if TK_NUM_LP >19
    TK_CHANNEL_19_LP,
#endif
#if TK_NUM_LP >20
    TK_CHANNEL_20_LP,
#endif
#if TK_NUM_LP >21
    TK_CHANNEL_21_LP,
#endif
#if TK_NUM_LP >22
    TK_CHANNEL_22_LP,
#endif
#if TK_NUM_LP >23
    TK_CHANNEL_23_LP,
#endif
#if TK_NUM_LP >24
    TK_CHANNEL_24_LP,
#endif
#if TK_NUM_LP >25
    TK_CHANNEL_25_LP,
#endif
#if TK_NUM_LP >26
    TK_CHANNEL_26_LP,
#endif
#if TK_NUM_LP >27
    TK_CHANNEL_27_LP,
#endif
#if TK_NUM_LP >28
    TK_CHANNEL_28_LP,
#endif
#if TK_NUM_LP >29
    TK_CHANNEL_29_LP,
#endif
#if TK_NUM_LP >30
    TK_CHANNEL_30_LP,
#endif
};

/* Threshold table for tk channel */
const uint16_t thd_table_lp[TK_NUM_LP] =
{
#if TK_NUM_LP >0
    TK_THD_CHANNEL_0_LP,
#endif
#if TK_NUM_LP >1
    TK_THD_CHANNEL_1_LP,
#endif
#if TK_NUM_LP >2
    TK_THD_CHANNEL_2_LP,
#endif
#if TK_NUM_LP >3
    TK_THD_CHANNEL_3_LP,
#endif
#if TK_NUM_LP >4
    TK_THD_CHANNEL_4_LP,
#endif
#if TK_NUM_LP >5
    TK_THD_CHANNEL_5_LP,
#endif
#if TK_NUM_LP >6
    TK_THD_CHANNEL_6_LP,
#endif
#if TK_NUM_LP >7
    TK_THD_CHANNEL_7_LP,
#endif
#if TK_NUM_LP >8
    TK_THD_CHANNEL_8_LP,
#endif
#if TK_NUM_LP >9
    TK_THD_CHANNEL_9_LP,
#endif
#if TK_NUM_LP >10
    TK_THD_CHANNEL_10_LP,
#endif
#if TK_NUM_LP >11
    TK_THD_CHANNEL_11_LP,
#endif
#if TK_NUM_LP >12
    TK_THD_CHANNEL_12_LP,
#endif
#if TK_NUM_LP >13
    TK_THD_CHANNEL_13_LP,
#endif
#if TK_NUM_LP >14
    TK_THD_CHANNEL_14_LP,
#endif
#if TK_NUM_LP >15
    TK_THD_CHANNEL_15_LP,
#endif
#if TK_NUM_LP >16
    TK_THD_CHANNEL_16_LP,
#endif
#if TK_NUM_LP >17
    TK_THD_CHANNEL_17_LP,
#endif
#if TK_NUM_LP >18
    TK_THD_CHANNEL_18_LP,
#endif
#if TK_NUM_LP >19
    TK_THD_CHANNEL_19_LP,
#endif
#if TK_NUM_LP >20
    TK_THD_CHANNEL_20_LP,
#endif
#if TK_NUM_LP >21
    TK_THD_CHANNEL_21_LP,
#endif
#if TK_NUM_LP >22
    TK_THD_CHANNEL_22_LP,
#endif
#if TK_NUM_LP >23
    TK_THD_CHANNEL_23_LP,
#endif
#if TK_NUM_LP >24
    TK_THD_CHANNEL_24_LP,
#endif
#if TK_NUM_LP >25
    TK_THD_CHANNEL_25_LP,
#endif
#if TK_NUM_LP >26
    TK_THD_CHANNEL_26_LP,
#endif
#if TK_NUM_LP >27
    TK_THD_CHANNEL_27_LP,
#endif
#if TK_NUM_LP >28
    TK_THD_CHANNEL_28_LP,
#endif
#if TK_NUM_LP >29
    TK_THD_CHANNEL_29_LP,
#endif
#if TK_NUM_LP >30
    TK_THD_CHANNEL_30_LP,
#endif
};

/**
  * @brief  LPtimer IRQ handler
  * @retval None
  */
void LP16T0_Handler(void)
{
    LPTIM0->IFC = 0x2;
    return;
}

/**
  * @brief  TK sleep initialization
  * @retval None
  */
void tk_lp_init(void)
{
    uint32_t i, k;
    uint32_t cnt = 0x4000000;

    __sleep_flag  = 0x3;
    __lp_base_cnt = 0x0;
    ald_cmu_perh_clock_config(CMU_PERH_LPTIM0, ENABLE);
    ald_mcu_irq_config(LP16T0_IRQn, 1, ENABLE);
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    SYSCFG_UNLOCK();
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, 0x1U << PMU_CR_LPM_POSS);
    SET_BIT(CMU->CLKENR, 0x10);
    MODIFY_REG(CMU->PERICR, CMU_PERICR_LPTIM0_MSK, 0x7U << CMU_PERICR_LPTIM0_POSS); /* ULRC */
    SYSCFG_LOCK();

    LPTIM0->UPDATE = 0x1;
    LPTIM0->CON0   = 0x0;
    LPTIM0->ARR    = 1000; /* 300ms, 1LSB = 0.1ms */
    LPTIM0->IER    = 0x2;
    LPTIM0->UPDATE = 0x0;

    while ((LPTIM0->SYNCSTAT & 0x4) && (--cnt));

    memset(lp_arr, 0x0, sizeof(tk_lp_arr_t) * TK_NUM_LP);

    for (i = 0; i < TK_NUM_LP; ++i)
    {
        for (k = 0; k < TK_NUM; ++k)
        {
            if (ch_table[k] == ch_table_lp[i])
                break;
        }

        __lp_map[i] = k;
    }

    return;
}

/**
  * @brief  TK sleep preparation
  * @retval None
  */
void tk_lp_prep(void)
{
    uint32_t i;

    if (tk_state)
        return;

    __sys_mode = SYS_MODE_LP;
    CLEAR_BIT(BS16T0->CON1, 0x1);
    TKS->CHEN = TK_CHANNEL_SEL_LP;
    TKS->IE   = 0x0;

    IWDT_UNLOCK();
    CLEAR_BIT(IWDT->CON, IWDT_CON_EN_MSK);
    IWDT_LOCK();
    /* Get average value in LowPower mode */
    __tk_lp_value_get();

    for (i = 0; i < TK_NUM_LP; ++i)
        lp_arr[i].avg = lp_arr[i].org;

    for (i = 0; i < TK_NUM_LP; ++i)
    {
        lp_arr[i].cnt = 0x0;
        lp_arr[i].flt = 0x0;
    }

    return;
}

/**
  * @brief  TK wake up preparation
  * @retval None
  */
void tk_wakeup_prep(void)
{
    __sys_mode = SYS_MODE_NORMAL;
    SET_BIT(__sleep_flag, 0x3);
    lp_idle_cnt = TK_LP_IDLE_TIME;
    TKS->CHEN = TK_CHANNEL_SEL;
    TKS->IE   = 0x17U;
    SET_BIT(BS16T0->CON1, 0x1);
    SET_BIT(TKS->CON0, 0x2); /* Enable TK scan */
    /* Start IWDT */
    ald_iwdt_start();
    return;
}

/**
  * @brief  Low power consumption
  * @retval None
  */
void tk_lp_enter(void)
{
    uint32_t cnt = 0x4000000;

    LPTIM0->CON1 = 0x1;

    while ((LPTIM0->SYNCSTAT & 0x2) && (--cnt));

    SET_BIT(LPTIM0->CON1, 0x2);

    while ((LPTIM0->SYNCSTAT & 0x2) && (--cnt));

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  TK process
  * @retval None
  */
void __tk_lp_value_get(void)
{
    uint8_t i, k;
    uint16_t val;

    for (i = 1; i < 4; ++i)
    {
        SET_BIT(TKS->CON0, 0x2);

        while (!(TKS->IF & 0x17));

        if (TKS->IF & 0x10)
        {
            for (k = 0; k < TK_NUM_LP; ++k)
            {
                val = TKS->CHRES[ch_table_lp[k] & 0x1F];

                if (i == 1)
                {
                    lp_arr[k].org = val;
                    lp_arr[k].max = lp_arr[k].org;
                    lp_arr[k].min = lp_arr[k].org;
                }
                else
                {
                    lp_arr[k].org += val;

                    if (val > lp_arr[k].max)
                        lp_arr[k].max = val;

                    if (val < lp_arr[k].min)
                        lp_arr[k].min = val;
                }
            }
        }
        else
        {
            i--;
        }

        TKS->IFC = 0x17;
    }

    for (i = 0; i < TK_NUM_LP; ++i)
        lp_arr[i].org = lp_arr[i].org - (lp_arr[i].max + lp_arr[i].min);

    return;
}

/**
  * @brief  Key handling procedure
  * @retval None
  */
void tk_lp_handler(void)
{
    uint8_t i, k, act;

    _lp_cnt++;
    act = 0x0;
    __tk_lp_value_get();

    for (i = 0; i < TK_NUM_LP; ++i)
    {
        if (lp_arr[i].org > (lp_arr[i].avg + thd_table_lp[i]))
        {
            act = 0x1;
        }
    }

    if (act == 0x0)
    {
        for (i = 0; i < TK_NUM_LP; ++i)
        {
            lp_arr[i].cnt = 0x0;
            lp_arr[i].flt = 0x0;
            lp_arr[i]._sum += lp_arr[i].org;
        }

        ++__lp_base_cnt;

        if (__lp_base_cnt == 0x8)
        {
            __lp_base_cnt = 0x0;

            for (i = 0; i < TK_NUM_LP; ++i)
            {
                lp_arr[i].avg  = ((lp_arr[i].avg + (lp_arr[i]._sum >> 3)) >> 1);
                lp_arr[i]._sum = 0x0;
            }
        }
    }
    else
    {
        __lp_base_cnt = 0x0;

        for (i = 0; i < TK_NUM_LP; ++i)
        {
            lp_arr[i]._sum = 0x0;
        }
    }

    if (act == 0x0)
        return;

    for (k = 0; k < TK_LP_DEBOUNCE_PRESS + 2; ++k)
    {
        __tk_lp_value_get();

        for (i = 0; i < TK_NUM_LP; ++i)
        {
            if (lp_arr[i].org > (lp_arr[i].avg + thd_table_lp[i]))
            {
                if (++lp_arr[i].cnt > TK_LP_DEBOUNCE_PRESS)
                {
                    lp_arr[i].cnt = 0x0;

                    if (++lp_arr[i].flt > TK_LP_FILTER_CNT)
                        SET_BIT(tk_state, (0x1 << __lp_map[i]));
                }
            }
            else
            {
                lp_arr[i].cnt = 0x0;
            }
        }
    }

    if (tk_state)
    {
        for (i = 0; i < TK_NUM_LP; ++i)
        {
            lp_arr[i].cnt = 0x0;
            lp_arr[i].flt = 0x0;
        }

        tk_wakeup_prep();
    }

    return;
}
#endif
