/**
  *********************************************************************************
  *
  * @file    tk_handle.c
  * @brief   TK Data Process
  *
  * @version V1.0
  * @date    06 Sep 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          06 Sep 2023     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  *********************************************************************************
  */
#include <string.h>
#include "ald_gpio.h"
#include "tk_config.h"
#include "tk_lib.h"
#include "tk_uart.h"
#include "tk_low_power.h"

/*ES32H0403LQ PIN-TK_ch Map   PA                      PB                               PA                    PB                PC   */
const uint8_t reg_port[31] = {0, 1, 2, 3, 4, 5, 6, 7, 0, 1, 2, 10, 11, 12, 13, 14, 15, 8, 9, 10, 11, 12, 15, 3, 4, 5, 6, 7, 9, 3, 4};
/*                        CH  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10, 11, 12, 13, 14, 15, 16,17,18, 19, 20, 21, 22,23,24,25,26,27,28,30,31 */

/* Channel table for use as tk channel */
const uint8_t ch_table[TK_NUM] =
{
#if TK_NUM >0
    TK_CHANNEL_0,
#endif
#if TK_NUM >1
    TK_CHANNEL_1,
#endif
#if TK_NUM >2
    TK_CHANNEL_2,
#endif
#if TK_NUM >3
    TK_CHANNEL_3,
#endif
#if TK_NUM >4
    TK_CHANNEL_4,
#endif
#if TK_NUM >5
    TK_CHANNEL_5,
#endif
#if TK_NUM >6
    TK_CHANNEL_6,
#endif
#if TK_NUM >7
    TK_CHANNEL_7,
#endif
#if TK_NUM >8
    TK_CHANNEL_8,
#endif
#if TK_NUM >9
    TK_CHANNEL_9,
#endif
#if TK_NUM >10
    TK_CHANNEL_10,
#endif
#if TK_NUM >11
    TK_CHANNEL_11,
#endif
#if TK_NUM >12
    TK_CHANNEL_12,
#endif
#if TK_NUM >13
    TK_CHANNEL_13,
#endif
#if TK_NUM >14
    TK_CHANNEL_14,
#endif
#if TK_NUM >15
    TK_CHANNEL_15,
#endif
#if TK_NUM >16
    TK_CHANNEL_16,
#endif
#if TK_NUM >17
    TK_CHANNEL_17,
#endif
#if TK_NUM >18
    TK_CHANNEL_18,
#endif
#if TK_NUM >19
    TK_CHANNEL_19,
#endif
#if TK_NUM >20
    TK_CHANNEL_20,
#endif
#if TK_NUM >21
    TK_CHANNEL_21,
#endif
#if TK_NUM >22
    TK_CHANNEL_22,
#endif
#if TK_NUM >23
    TK_CHANNEL_23,
#endif
#if TK_NUM >24
    TK_CHANNEL_24,
#endif
#if TK_NUM >25
    TK_CHANNEL_25,
#endif
#if TK_NUM >26
    TK_CHANNEL_26,
#endif
#if TK_NUM >27
    TK_CHANNEL_27,
#endif
#if TK_NUM >28
    TK_CHANNEL_28,
#endif
#if TK_NUM >29
    TK_CHANNEL_29,
#endif
#if TK_NUM >30
    TK_CHANNEL_30,
#endif
#if TK_NUM >31
    TK_CHANNEL_31,
#endif
};

/* Threshold table for tk channel */
const uint16_t thd_table[TK_NUM] =
{
#if TK_NUM >0
    TK_THD_CHANNEL_0,
#endif
#if TK_NUM >1
    TK_THD_CHANNEL_1,
#endif
#if TK_NUM >2
    TK_THD_CHANNEL_2,
#endif
#if TK_NUM >3
    TK_THD_CHANNEL_3,
#endif
#if TK_NUM >4
    TK_THD_CHANNEL_4,
#endif
#if TK_NUM >5
    TK_THD_CHANNEL_5,
#endif
#if TK_NUM >6
    TK_THD_CHANNEL_6,
#endif
#if TK_NUM >7
    TK_THD_CHANNEL_7,
#endif
#if TK_NUM >8
    TK_THD_CHANNEL_8,
#endif
#if TK_NUM >9
    TK_THD_CHANNEL_9,
#endif
#if TK_NUM >10
    TK_THD_CHANNEL_10,
#endif
#if TK_NUM >11
    TK_THD_CHANNEL_11,
#endif
#if TK_NUM >12
    TK_THD_CHANNEL_12,
#endif
#if TK_NUM >13
    TK_THD_CHANNEL_13,
#endif
#if TK_NUM >14
    TK_THD_CHANNEL_14,
#endif
#if TK_NUM >15
    TK_THD_CHANNEL_15,
#endif
#if TK_NUM >16
    TK_THD_CHANNEL_16,
#endif
#if TK_NUM >17
    TK_THD_CHANNEL_17,
#endif
#if TK_NUM >18
    TK_THD_CHANNEL_18,
#endif
#if TK_NUM >19
    TK_THD_CHANNEL_19,
#endif
#if TK_NUM >20
    TK_THD_CHANNEL_20,
#endif
#if TK_NUM >21
    TK_THD_CHANNEL_21,
#endif
#if TK_NUM >22
    TK_THD_CHANNEL_22,
#endif
#if TK_NUM >23
    TK_THD_CHANNEL_23,
#endif
#if TK_NUM >24
    TK_THD_CHANNEL_24,
#endif
#if TK_NUM >25
    TK_THD_CHANNEL_25,
#endif
#if TK_NUM >26
    TK_THD_CHANNEL_26,
#endif
#if TK_NUM >27
    TK_THD_CHANNEL_27,
#endif
#if TK_NUM >28
    TK_THD_CHANNEL_28,
#endif
#if TK_NUM >29
    TK_THD_CHANNEL_29,
#endif
#if TK_NUM >30
    TK_THD_CHANNEL_30,
#endif
#if TK_NUM >31
    TK_THD_CHANNEL_31,
#endif
};

/* Threshold * 9 / 10 */
const uint16_t thd_table_9_10[TK_NUM] =
{
#if TK_NUM >0
    TK_THD_CHANNEL_0 * 9 / 10,
#endif
#if TK_NUM >1
    TK_THD_CHANNEL_1 * 9 / 10,
#endif
#if TK_NUM >2
    TK_THD_CHANNEL_2 * 9 / 10,
#endif
#if TK_NUM >3
    TK_THD_CHANNEL_3 * 9 / 10,
#endif
#if TK_NUM >4
    TK_THD_CHANNEL_4 * 9 / 10,
#endif
#if TK_NUM >5
    TK_THD_CHANNEL_5 * 9 / 10,
#endif
#if TK_NUM >6
    TK_THD_CHANNEL_6 * 9 / 10,
#endif
#if TK_NUM >7
    TK_THD_CHANNEL_7 * 9 / 10,
#endif
#if TK_NUM >8
    TK_THD_CHANNEL_8 * 9 / 10,
#endif
#if TK_NUM >9
    TK_THD_CHANNEL_9 * 9 / 10,
#endif
#if TK_NUM >10
    TK_THD_CHANNEL_10 * 9 / 10,
#endif
#if TK_NUM >11
    TK_THD_CHANNEL_11 * 9 / 10,
#endif
#if TK_NUM >12
    TK_THD_CHANNEL_12 * 9 / 10,
#endif
#if TK_NUM >13
    TK_THD_CHANNEL_13 * 9 / 10,
#endif
#if TK_NUM >14
    TK_THD_CHANNEL_14 * 9 / 10,
#endif
#if TK_NUM >15
    TK_THD_CHANNEL_15 * 9 / 10,
#endif
#if TK_NUM >16
    TK_THD_CHANNEL_16 * 9 / 10,
#endif
#if TK_NUM >17
    TK_THD_CHANNEL_17 * 9 / 10,
#endif
#if TK_NUM >18
    TK_THD_CHANNEL_18 * 9 / 10,
#endif
#if TK_NUM >19
    TK_THD_CHANNEL_19 * 9 / 10,
#endif
#if TK_NUM >20
    TK_THD_CHANNEL_20 * 9 / 10,
#endif
#if TK_NUM >21
    TK_THD_CHANNEL_21 * 9 / 10,
#endif
#if TK_NUM >22
    TK_THD_CHANNEL_22 * 9 / 10,
#endif
#if TK_NUM >23
    TK_THD_CHANNEL_23 * 9 / 10,
#endif
#if TK_NUM >24
    TK_THD_CHANNEL_24 * 9 / 10,
#endif
#if TK_NUM >25
    TK_THD_CHANNEL_25 * 9 / 10,
#endif
#if TK_NUM >26
    TK_THD_CHANNEL_26 * 9 / 10,
#endif
#if TK_NUM >27
    TK_THD_CHANNEL_27 * 9 / 10,
#endif
#if TK_NUM >28
    TK_THD_CHANNEL_28 * 9 / 10,
#endif
#if TK_NUM >29
    TK_THD_CHANNEL_29 * 9 / 10,
#endif
#if TK_NUM >30
    TK_THD_CHANNEL_30 * 9 / 10,
#endif
#if TK_NUM >31
    TK_THD_CHANNEL_31 * 9 / 10,
#endif
};

uint32_t tk_jitter_thd_lv0, tk_jitter_thd_lv1, tk_jitter_thd_lv2;
uint16_t tk_press_timeout, tk_base_up_time;
uint8_t tk_anti_motor, tk_adaptive, tk_enhance_filter, timer_10ms, tk_ch_num, tk_sample_perscan, tk_debounce_press, tk_debounce_release;
tk_arr_t tk_arr[TK_NUM];
uint8_t __sys_mode;

#if TK_LOWPOWER_MODE == ON
    uint16_t lp_idle_cnt;
#endif

/**
  * @brief  TK IRQ handler
  * @retval None
  */
void TKS_Handler(void)
{
    if (TKS->IF & 0x10)
    {
        tk_value_get = 1;
    }
    else
    {
        tk_value_get = 0;
    }

    TKS->IFC = 0x1FU;

    if (tk_value_get == 0)
    {
        SET_BIT(TKS->CON0, 0x2); /* Enable TK scan */
    }

    return;
}

/**
  * @brief  Touch channel initialization.
  * @retval None
  */
void tk_pin(void)
{
    gpio_init_t x;
    unsigned char i, j, k;

    /* Initialize GPIO_PIN_0 sa output */
    x.mode = GPIO_MODE_CLOSE;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_FLOATING;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.func = GPIO_FUNC_0;

    for (i = 0; i < tk_ch_num; i++)
    {
        j = (ch_table[i] & 0x1F);
        j = reg_port[j];
        k = (ch_table[i] & 0xE0);

        if (k == PA_PORT)             // PA
        {
            ald_gpio_init(GPIOA, (uint16_t)(1 << j), &x);
        }
        else if (k == PB_PORT)        // PB
        {
            ald_gpio_init(GPIOB, (uint16_t)(1 << j), &x);
        }
        else if (k == PC_PORT)        // PC
        {
            ald_gpio_init(GPIOC, (uint16_t)(1 << j), &x);
        }
    }

    //Cx
    ald_gpio_init(GPIOB, GPIO_PIN_8, &x);
}

/**
  * @brief  Initializate TK moudle.
  * @retval None
  */
void tk_init(void)
{
    uint8_t i;

    tk_state        = 0x0;
    tk_state_bk     = 0x0;
    base_up_cnt     = 0x0;
    lock_avg_cnt    = 0x0;
    filter_cnt      = 0x0;
    tkscan_cnt      = 0x0;
    __tk_jt0_cnt0   = 0x0;
    __tk_jt0_cnt1   = 0x0;
    __sys_mode      = SYS_MODE_NORMAL;

    tk_jitter_thd_lv0 = 0x0;

    for (i = 0; i < TK_NUM; ++i)
    {
        memset(&tk_arr[i], 0x0, sizeof(tk_arr_t));
        tk_jitter_thd_lv0 += thd_table[i];
    }

    tk_jitter_thd_lv2 = TK_JITTER_THD_LV2;
    tk_jitter_thd_lv1 = TK_JITTER_THD_LV1;
    tk_jitter_thd_lv0 = TK_JITTER_THD_LV0;
    tk_init_state = FIRST_POWER_UP;
    tk_press_timeout = TK_PRESS_TIMEOUT;
    tk_debounce_press = TK_DEBOUNCE_PRESS;
    tk_debounce_release = TK_DEBOUNCE_RELEASE;
    tk_sample_perscan = TK_SAMPLE_PERSCAN;
    tk_base_up_time = TK_BASE_UP_TIME;
    tk_ch_num = TK_NUM;

#ifdef TK_ENHANCE_FILTER
    tk_enhance_filter = 1;
#else
    tk_enhance_filter = 0;
#endif
#ifdef TK_THRESHOLD_ADAPTIVE
    tk_adaptive = 1;
#else
    tk_adaptive = 0;
#endif
#ifdef TK_ANTI_MOTOR
    tk_anti_motor = 1;
#else
    tk_anti_motor = 0;
#endif

    /* Enable the TK clock */
    ald_cmu_tks_clock_select(CMU_TKS_SEL_HRC);
    ald_cmu_perh_clock_config(CMU_PERH_TKS, ENABLE);
    ald_mcu_irq_config(TKS_IRQn, 1, ENABLE);

    return;
}

/**
  * @brief  Initializate TK moudle.
  * @retval None
  */
void tk_reg_config(void)
{
    TKS->CHEN  = TK_CHANNEL_SEL;
    TKS->CON0  = TK_REG_CON0;
    TKS->CON1  = TK_REG_CON1_REF;
    TKS->GAIN  = TK_REG_GAIN;
    TKS->SFJTR = TK_REG_SFJTR;
    TKS->IFC   = 0x1FU;
    TKS->IE    = 0x17U;

    tk_value_get = 0;
    tk_init_state = DEFAULT;
}

/**
  * @brief  Touch jitter.
  * @retval None
  */
void tk_jitter(void)
{
#if AUTO_JITTER_SOFTWARE == ON


    if (lock_avg_cnt)
    {
        if (TKS->CON0 == TK_REG_CON0_BK)

        {
            TKS->CON0 = TK_REG_CON0;
        }
        else
        {
            TKS->CON0 = TK_REG_CON0_BK;
        }
    }
    else
    {
        TKS->CON0 = TK_REG_CON0;
    }

#endif
#if AUTO_JITTER_HARDWARE == ON

    if (lock_avg_cnt)
    {
        TKFJCTL = 0x02;
    }
    else
    {
        TKFJCTL = 0x00;
    }

#endif
    tk_value_get = 0;
    tk_init_state = DEFAULT;
}

/**
  * @brief  Check the time flag.
  * @retval None
  */
void timer_check(void)
{
    if (timer_10ms < 1)
        return;

    timer_10ms = 0;

    tk_base_up_lock_cnt();
#if TK_LOWPOWER_MODE == ON
    lp_idle_cnt = lp_idle_cnt > 0 ? lp_idle_cnt - 1 : lp_idle_cnt;

    if (lp_idle_cnt == 0)
    {
        CLEAR_BIT(__sleep_flag, 0x1);
    }

#endif

#if TK_PRESS_TIMEOUT != 0
    tk_press_to();
#endif

#if TK_UART_FUNC == ON

    if (Tx_timer)
    {
        Tx_timer++;

        if (Tx_timer > 100)
        {
            Tx_timer = 0;
            g_tp = 0;
            Uart_ready = 1;
        }       //Tx send 1s timeout send Reset
    }

    if (Rx_timer)
    {
        Rx_timer++;

        if (Rx_timer > 100)
        {
#if TK_UART_TYPE == TKM_UART
            ald_uart_recv_by_it(&h_uart, &g_com, 1);
#endif
#if TK_UART_TYPE == TKM_USART
            ald_usart_recv_by_it(&h_usart0, &g_com, 1);
#endif
#if TK_UART_TYPE == TKM_LPUART
            ald_lpuart_recv_frame_by_it(&h_lpuart, &g_com, 1, 40);
#endif
            Uart_ready = 0;
            Rx_timer = 1;
        }
    }

#endif
    return;
}

/**
  * @brief  Touch key processing.
  * @retval None
  */
void bsp_tk_handler(void)
{
    if (__sys_mode == SYS_MODE_NORMAL)
    {
        tk_service();
        timer_check();
#if TK_LOWPOWER_MODE == ON

        if (tk_state)
        {
            SET_BIT(__sleep_flag, 0x1);
            lp_idle_cnt = TK_LP_IDLE_TIME;
        }

        if (!(__sleep_flag & 0x1))
        {
            CLEAR_BIT(__sleep_flag, 0x2);
            tk_lp_prep();
        }

#endif
#if TK_UART_FUNC == ON

        if (Uart_ready)
        {
            Uart_ready = 0;
            Rx_timer = 1;
#if TK_UART_TYPE == TKM_UART
            ald_uart_recv_by_it(&h_uart, &g_com, 1);
#endif
#if TK_UART_TYPE == TKM_USART
            ald_usart_recv_by_it(&h_usart0, &g_com, 1);
#endif
#if TK_UART_TYPE == TKM_LPUART
            ald_lpuart_recv_frame_by_it(&h_lpuart, &g_com, 1, 40);
#endif
        }

#endif
    }

#if TK_LOWPOWER_MODE == ON

    if (__sys_mode == SYS_MODE_LP)
    {
        tk_lp_handler();

        if (!(__sleep_flag))
        {
            tk_lp_enter();
        }
    }

#endif
    return;
}

/**
  * @brief  Touch initialization.
  * @retval None
  */
void bsp_tk_init(void)
{
    tk_pin();
    tk_init();
#if TK_UART_FUNC == ON
    tk_uart_init();
#endif
#if TK_LOWPOWER_MODE == ON
    lp_idle_cnt = TK_LP_IDLE_TIME;
    tk_lp_init();
#endif
    return;
}
