/**
  *********************************************************************************
  *
  * @file    ald_pmu.c
  * @brief   PMU module driver.
  *
  * @version V1.2
  * @date    24 Jun 2022
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jun 2020     AE Team         The first version
  *          21 Mar 2022     AE Team         add register PMU->BKPCR0 unlock code
  *          24 Jun 2022     AE Team         modified function ald_pmu_shutoff_enter
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "ald_conf.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup PMU PMU
  * @brief PMU module driver
  * @{
  */
#ifdef ALD_PMU


/** @defgroup PMU_Private_Functions PMU Private Functions
  * @{
  */

/**
  * @brief  PMU module interrupt handler
  * @retval None
  */
void ald_lvd_irq_handler(void)
{
    SYSCFG_UNLOCK();
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
    SYSCFG_LOCK();

    return;
}
/**
  * @}
  */

/** @defgroup PMU_Public_Functions PMU Public Functions
  * @{
  */

/** @addtogroup PMU_Public_Functions_Group1 Low Power Mode
  * @brief Low power mode select functions
  *
  * @verbatim
  ==============================================================================
              ##### Low power mode select functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Enter stop1 mode.
      (+) Enter stop2 mode.
      (+) Get wakeup status.
      (+) Clear wakeup status.

    @endverbatim
  * @{
  */

/**
  * @brief  Enter stop1 mode
  * @retval None
  */
void ald_pmu_stop1_enter(void)
{
    SYSCFG_UNLOCK();
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, PMU_LP_STOP1 << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter stop2 mode
  * @retval None
  */
void ald_pmu_stop2_enter(void)
{
    SYSCFG_UNLOCK();
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, PMU_LP_STOP2 << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter standby mode
  * @retval None
  */
void ald_pmu_standby_enter(pmu_wakeup_pin_t pin, pmu_wakeup_level_t levle)
{
    uint32_t tmp = 0x5A00;

    assert_param(IS_PMU_WAKEUP_PIN(pin));
    assert_param(IS_PMU_WAKEUP_LEVEL(levle));

    SET_BIT(tmp, PMU_BKPCR0_WKPEN_MSK | PMU_BKPCR0_STBWKEN_MSK);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPS_MSK, pin << PMU_BKPCR0_WKPS_POSS);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPL_MSK, levle << PMU_BKPCR0_WKPL_POS);

    SYSCFG_UNLOCK();
    PMU->BKPCR0 = tmp;
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, PMU_LP_STANDBY << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter shutoff mode
  * @retval None
  */
void ald_pmu_shutoff_enter(pmu_wakeup_pin_t pin, pmu_wakeup_level_t levle)
{
    uint32_t tmp = 0x5A00;

    assert_param(IS_PMU_WAKEUP_PIN(pin));
    assert_param(IS_PMU_WAKEUP_LEVEL(levle));

    SET_BIT(tmp, PMU_BKPCR0_WKPEN_MSK | PMU_BKPCR0_STBWKEN_MSK);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPS_MSK, pin << PMU_BKPCR0_WKPS_POSS);
    MODIFY_REG(tmp, PMU_BKPCR0_WKPL_MSK, levle << PMU_BKPCR0_WKPL_POS);

    SYSCFG_UNLOCK();
    PMU->BKPCR0 = tmp;
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, PMU_LP_SHUTOFF << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Configures low power mode. The system clock must
  *         be less than 2MHz. Such as: LOSC or LRC.
  * @param  vol: LDO output voltage select in low power mode.
  * @param  state: New state, ENABLE/DISABLE;
  * @retval None
  */
void ald_pmu_lprun_config(pmu_ldo_lpmode_output_t vol, type_func_t state)
{
    assert_param(IS_FUNC_STATE(state));
    SYSCFG_UNLOCK();

    if (state)
    {
        assert_param(IS_PMU_LDO_LPMODE_OUTPUT(vol));

        MODIFY_REG(PMU->CR, PMU_CR_LPVS_MSK, vol << PMU_CR_LPVS_POSS);
        MODIFY_REG(PMU->CR, PMU_CR_LPVSTK_MSK, vol << PMU_CR_LPVSTK_POSS);
        SET_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
    }
    else
    {
        CLEAR_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Set wakeup time.
  * @param  time: Time that will be set.
  * @retval None
  */
void ald_pmu_wakup_set(uint16_t time)
{
    time = time & 0xFFF;

    SYSCFG_UNLOCK();
    PMU->TWUR = time;
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Write data into backup ram.
  * @param  idx: Index of backup word.
  * @param  value: Value which will be written to backup ram.
  * @retval None
  */
void ald_pmu_bkp_write_ram(uint8_t idx, uint32_t value)
{
    assert_param(IS_PMU_BKP_RAM_IDX(idx));

    SYSCFG_UNLOCK();
    WRITE_REG(PMU->BKPREG[idx], value);
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Read data from backup ram.
  * @param  idx: Index of backup word.
  * @retval The data.
  */
uint32_t ald_pmu_bkp_read_ram(uint8_t idx)
{
    assert_param(IS_PMU_BKP_RAM_IDX(idx));

    return READ_REG(PMU->BKPREG[idx]);
}

/**
  * @brief  Get wakup status.
  * @param  sr: Status bit.
  * @retval Status.
  */
flag_status_t ald_pmu_get_status(pmu_status_t sr)
{
    assert_param(IS_PMU_STATUS(sr));

    if (READ_BIT(PMU->SR, sr))
        return SET;

    return RESET;
}

/**
  * @brief  Clear wakup status.
  * @param  sr: Status bit.
  * @retval None
  */
void ald_pmu_clear_status(pmu_status_t sr)
{
    assert_param(IS_PMU_STATUS(sr));
    SYSCFG_UNLOCK();

    if (sr == PMU_SR_WUF)
        SET_BIT(PMU->CR, PMU_CR_CWUF_MSK);
    else if (sr == PMU_SR_STANDBY)
        SET_BIT(PMU->CR, PMU_CR_CSTANDBYF_MSK);
    else
        SET_BIT(PMU->CR, PMU_CR_CSHUTOFF_MSK);

    SYSCFG_LOCK();
    return;
}
/**
  * @}
  */

/** @addtogroup PMU_Public_Functions_Group2 LVD Configure
  * @brief LVD configure functions
  *
  * @verbatim
  ==============================================================================
              ##### LVD configure functions #####
  ==============================================================================
    [..]  This section provides functions allowing to:
      (+) Configure lvd parameters.
      (+) Interrupt callback function.

    @endverbatim
  * @{
  */

/**
  * @brief  Configure lvd using specified parameters.
  * @param  sel: LVD threshold voltage.
  * @param  mode: LVD trigger mode.
  * @param  state: New state, ENABLE/DISABLE;
  * @retval None
  */
void ald_pmu_lvd_config(pmu_lvd_voltage_sel_t sel, pmu_lvd_trigger_mode_t mode, type_func_t state)
{
    assert_param(IS_FUNC_STATE(state));
    SYSCFG_UNLOCK();

    if (state)
    {
        assert_param(IS_PMU_LVD_VOL_SEL(sel));
        assert_param(IS_PMU_LVD_TRIGGER_MODE(mode));

        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, sel << PMU_LVDCR_LVDS_POSS);
        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDIFS_MSK, mode << PMU_LVDCR_LVDIFS_POSS);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }
    else
    {
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }

    SYSCFG_LOCK();
    return;
}
/**
  * @}
  */


/**
  * @}
  */
#endif /* ALD_PMU */
/**
  * @}
  */

/**
  * @}
  */
