/**
  *********************************************************************************
  *
  * @file    customer.c
  * @brief   Customer Process
  *
  * @version V1.2
  * @date    18 Oct 2023
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          2 Dec 2021      AE Team         The first version
  *          29 Dec 2022     AE Team         Add serial port options
  *          18 Oct 2023     AE Team         Modified the TK boot mode in low power mode
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  *********************************************************************************
  */

#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_timer.h"
#include "ald_rmu.h"
#include "timer.h"
#include "tk_uart.h"
#include "tkm_config.h"
#include "tk_handle.h"
#include "tk_low_power.h"
#include "customer.h"
#include "display.h"
#include "beep.h"

/**
  * @brief  key press handling function.
  * @retval None
  */
void Customer_Keystatejustpress(void)
{
    //TK_state was just freshed  you can use the  parameter as you want
    if (TK_state > TK_state_bk)
    {
        TK_state_bk = TK_state;
        FMQ_counter = BEEPTIME;  //FMQ 250MS
        Beep_On();

        Press_counter++;

        if (Press_counter > 99)
        {
            Press_counter = 0;
        }

        dispBuf[0] = TK_state & 0x0F;
        dispBuf[1] = (TK_state >> 4) & 0x0F;
        dispBuf[2] = Press_counter / 10; //(TK_state >> 8) & 0x0F;
        dispBuf[3] = Press_counter % 10;

        ald_lcd_write(&h_lcd, 2, dispNumTab[dispBuf[0]]);
        ald_lcd_write(&h_lcd, 1, dispNumTab[dispBuf[1]]);
        ald_lcd_write(&h_lcd, 0, dispNumTab[dispBuf[2]]);
        ald_lcd_write(&h_lcd, 3, dispNumTab[dispBuf[3]]);
    }
}

/**
  * @brief  key release handler.
  * @retval None
  */
void Customer_Keystatejustrelease(void)
{
    //TK_state was just freshed  you can use the  parameter as you want
    if (TK_state < TK_state_bk)
    {
        TK_state_bk = TK_state;
        dispBuf[0] = TK_state & 0x0F;
        dispBuf[1] = (TK_state >> 4) & 0x0F;

        ald_lcd_write(&h_lcd, 2, dispNumTab[dispBuf[0]]);
        ald_lcd_write(&h_lcd, 1, dispNumTab[dispBuf[1]]);
    }
}

#if TK_Lowpowermode == ON

/**
  * @brief  sleep preprocessor function.
  * @retval None
  */
void Customer_prepareforsleepmode_init(void)
{
    TK_Sleep_cnt = 0;
    g_tk_done_if = 0;
//    gpio_init_t x;
//  memset(&x, 0x0, sizeof(x));

//    /* Initialize beep pin */
//  x.mode = GPIO_MODE_CLOSE;
//  x.odos = GPIO_PUSH_PULL;
//  x.pupd = GPIO_FLOATING;
//  x.flt  = GPIO_FILTER_DISABLE;
//  x.type = GPIO_TYPE_TTL;
//  x.nodrv = GPIO_OUT_DRIVE_6;
//  x.podrv = GPIO_OUT_DRIVE_6;
//  x.func = GPIO_FUNC_0;

//  ald_gpio_init(GPIOA, GPIO_PIN_4, &x);
//
//  /* Lights down */
//  ald_gpio_write_pin(GPIOB, GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_10|GPIO_PIN_11|GPIO_PIN_12|GPIO_PIN_13|GPIO_PIN_14|GPIO_PIN_15, 0);

//  ald_gpio_write_pin(GPIOA, GPIO_PIN_10|GPIO_PIN_9|GPIO_PIN_8, 1);
//  ald_gpio_write_pin(GPIOB, GPIO_PIN_9, 1);

//  /* Beep Off */
//  ald_gpio_write_pin(GPIOA, GPIO_PIN_4, 0);

    ald_gpio_func_default(GPIOA);
    ald_gpio_func_default(GPIOB);
    /* disable the  clock */
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, DISABLE);

    ald_lcd_cmd(&h_lcd, DISABLE);
    ald_lcd_led_init(&h_lcd, DISABLE);
    ald_cmu_perh_clock_config(CMU_PERH_LCD, DISABLE);

    ald_timer_pwm_stop(&h_tim0, TIMER_CHANNEL_2);
    ald_cmu_perh_clock_config(CMU_PERH_GP16C4T0, DISABLE);

#if TK_UARToutput_function == ON
    g_tp = 0;
#if TK_UART_TYPE == TKM_UART
    ald_mcu_irq_config(UART0_IRQn, 1, DISABLE);
    ald_uart_reset(&h_uart_tk);
    ald_cmu_perh_clock_config(CMU_PERH_UART0, DISABLE);
#endif
#if TK_UART_TYPE == TKM_USART
    ald_mcu_irq_config(USART0_IRQn, 1, DISABLE);
    ald_usart_reset(&h_usart0_tk);
    ald_cmu_perh_clock_config(CMU_PERH_USART0, DISABLE);
#endif
#if TK_UART_TYPE == TKM_LPUART
    ald_mcu_irq_config(LPUART0_IRQn, 1, DISABLE);
    ald_lpuart_reset(&h_lpuart_tk);
    ald_cmu_perh_clock_config(CMU_PERH_LPUART0, DISABLE);
#endif
#endif

    TKS->CON0 = 0x0;
    TKS->CON1 = 0x0;
    TKS->CHEN = 0x0;
    TKS->GAIN = 0x0;
    TKS->IE = 0x0;
    TKS->IFC = 0xFF;
    TKS->SFJTR = 0x0;
    ald_cmu_perh_clock_config(CMU_PERH_TKS, DISABLE);

    ald_mcu_irq_config(BS16T0_IRQn, 2, DISABLE);
    ald_timer_interrupt_config(&h_tim1, TIMER_IT_UPDATE, DISABLE);
    ald_timer_base_reset(&h_tim1);
    ald_cmu_perh_clock_config(CMU_PERH_BS16T0, DISABLE);

    TKS_init();

    LPTIM_init();

    SYSCFG_UNLOCK();
    SET_BIT(PMU->CR, PMU_CR_FSTOP_MSK);
    SYSCFG_LOCK();
}

/**
  * @brief  wake up preprocessor function.
  * @retval None
  */
void Customer_wakeupfromsleepmode_init(void)
{
#if (Jitter_Function == ON)||(TK_LP_dataprocess_mode == Poll)
    uint8_t i = 0;
#endif

    ald_lptim_reset(&h_lptim);
    ald_cmu_perh_clock_config(CMU_PERH_LPTIM0, DISABLE);

    TKS->CON0 = 0x0;
    TKS->CON1 = 0x0;
    TKS->CHEN = 0x0;
    TKS->GAIN = 0x0;
    TKS->IE = 0x0;
    TKS->IFC = 0xFF;
    TKS->SFJTR = 0x0;

#if TK_UARToutput_function == ON
#if TK_UART_TYPE == TKM_UART
    UART_Init_tk();
#endif
#if TK_UART_TYPE == TKM_USART
    USART_Init_tk();
#endif
#if TK_UART_TYPE == TKM_LPUART
    LPUART_Init_tk();
#endif
#endif

    LCD_Init();
    Beep_Init();

    Tkscan_numth        = 0;
    Tkscan_sampcounter  = 0;
    Opr_state           = 0x8000;//Enable scan
    TK_state            = 0;
    TK_Base_percounter  = 0;
    TK_value_get        = 0;
    Timer_10ms          = 0;
    Lock_averageconter  = 0;
    Timer_counter = 0;

    g_wakeup_flag = 1;
    TK_Sleep_cnt = 0;
    TK_mode1_nopresstimer = 0;
#if TK_LP_dataprocess_mode == Merge
    TK_md2average_backup = 0;
    sleep_first_data = 0;
    sleep_jitter_data = 0;
    lp_base_err_cnt = 0;
#endif
#if TK_LP_dataprocess_mode == Poll

    for (i = 0; i < TK_NUM_LP; i++)
    {
        TK_md2average_backup[i] = 0;
        sleep_first_data[i] = 0;
        sleep_jitter_data[i] = 0;
        lp_base_err_cnt[i] = 0;
    }

#endif
    Opr_state |= 0x0020;

#if Jitter_Function == ON

    for (i = 0; i < TK_NUM; i++)   //Clear all
    {
        TK_Jitter_tolerance[i] = 0;
    }

#endif

    /* Enable the TK clock */
    ald_cmu_tks_clock_select(CMU_TKS_SEL_HRC);
    ald_cmu_perh_clock_config(CMU_PERH_TKS, ENABLE);

    TKS->CHEN = TK_Channel_Sel;
    TKS->CON0 = TK_reg_CON0;        //24Ƶ  1/2ռձ  ΪŴϵԳŵֵ
    TKS->CON1 = TK_reg_CON1;        //CpCx  Cxŵ256ʱ  2.5V
    TKS->GAIN = TK_Amplifi_setting;
    TKS->SFJTR = TK_reg_SFJTR;      //΢Ƶ
//  TKS->IE = 0x10;                 //ʹж
    TKS->IFC = 0x1F;                //жϱ־
    /* Initialize BS16T0 */
    BS16T0_Init();
#if TK_Guardsensor_output == ON
    TKS->CON0 |= 0x00002000;
#endif
#if Auto_Jitter_hardware == ON
    TKS->SFJTR |= 0x01000000;
#endif
    TKS->CON0 |= 0x02;
}

#endif
