/**
  *********************************************************************************
  *
  * @file    beep.c
  * @brief   Beep Process
  *
  * @version V1.0
  * @date    2 Dec 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          2 Dec 2021      AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  *********************************************************************************
  */

#include <string.h>
#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_timer.h"
#include "display.h"
#include "beep.h"

uint16_t FMQ_counter;
timer_handle_t h_tim0;

/**
  * @brief  Initializate beep pin.
  * @retval None
  */
void beep_pin_init(void)
{
    gpio_init_t x;
    memset(&x, 0x0, sizeof(x));

    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);

    /* Initialize beep pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.func = GPIO_FUNC_2;

    ald_gpio_init(GPIOA, GPIO_PIN_4, &x);
}

/**
  * @brief  Initializate beep.
  * @retval None
  */
void Beep_Init(void)
{
    uint32_t __value;

    timer_clock_config_t tim0_clock;
    timer_oc_init_t tim0_ocinit;

    FMQ_counter = BEEPTIME;

    memset(&h_tim0, 0x0, sizeof(h_tim0));
    memset(&tim0_clock, 0x0, sizeof(tim0_clock));
    memset(&tim0_ocinit, 0x0, sizeof(tim0_ocinit));
    /* Initialize pin */
    beep_pin_init();

    ald_cmu_perh_clock_config(CMU_PERH_GP16C4T0, ENABLE);

    /* Compute the prescaler value to have TIM0 counter clock equal to 2000000 Hz */
    __value = (uint32_t)((ald_cmu_get_pclk1_clock()) / 2000000) - 1;

    /* Initialize GP16C4T0 */
    h_tim0.perh              = GP16C4T0;
    h_tim0.init.prescaler    = __value;
    h_tim0.init.mode         = TIMER_CNT_MODE_UP;
    h_tim0.init.period       = PERIOD_VALUE;
    h_tim0.init.clk_div      = TIMER_CLOCK_DIV1;
    h_tim0.init.re_cnt       = 0;
    ald_timer_pwm_init(&h_tim0);

    /* Initialize clock source */
    tim0_clock.source = TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim0, &tim0_clock);

    /* Common configuration for all channels */
    tim0_ocinit.oc_mode      = TIMER_OC_MODE_PWM1;
    tim0_ocinit.oc_polarity  = TIMER_OC_POLARITY_HIGH;
    tim0_ocinit.oc_fast_en   = DISABLE;
    tim0_ocinit.ocn_polarity = TIMER_OCN_POLARITY_HIGH;
    tim0_ocinit.ocn_idle     = TIMER_OCN_IDLE_RESET;
    tim0_ocinit.oc_idle      = TIMER_OC_IDLE_RESET;

    /* Set the pulse value for channel 2 */
    tim0_ocinit.pulse = PULSE2_VALUE;
    ald_timer_oc_config_channel(&h_tim0, &tim0_ocinit, TIMER_CHANNEL_2);
}

/**
  * @brief  beep on.
  * @retval None
  */
void Beep_On(void)
{
    /* Start input pwm from tim0 channel 2 */
    ald_timer_oc_start(&h_tim0, TIMER_CHANNEL_2);
}

/**
  * @brief  beep off.
  * @retval None
  */
void Beep_Off(void)
{
    /* Stop input pwm from tim0 channel 2 */
    ald_timer_oc_stop(&h_tim0, TIMER_CHANNEL_2);
}
