/**
  *********************************************************************************
  *
  * @file    tk_low_power.c
  * @brief   Low Power Process
  *
  * @version V1.1
  * @date    29 Dec 2022
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          2 Dec 2021      AE Team         The first version
  *          29 Dec 2022     AE Team         Add serial port options
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  *********************************************************************************
  */

#include <string.h>
#include "ald_pmu.h"
#include "tkm_config.h"
#include "tk_handle_S&W.h"
#include "tk_low_power.h"
#include "uart.h"
#include "customer.h"

#if TK_Lowpowermode == ON

lptim_handle_t h_lptim;
uint8_t g_tk_done_if;
uint8_t TK_Sleep_cnt;
uint8_t g_wakeup_flag;
#if TK_LP_dataprocess_mode == Merge
    uint16_t TK_md2average_backup;
#endif
#if TK_LP_dataprocess_mode == Poll
    uint16_t TK_md2average_backup[TK_NUM];
#endif
uint16_t TK_mode1_nopresstimer;

/**
  * @brief  TK sleep mode config.
  * @retval None
  */
void TKS_init(void)
{
    ald_cmu_tks_clock_select(CMU_TKS_SEL_HRC);
    ald_cmu_perh_clock_config(CMU_PERH_TKS, ENABLE);
    ald_mcu_irq_config(TKS_IRQn, 1, ENABLE);

#if TK_LP_dataprocess_mode  == Merge
    TKS->CON0 = TK_reg_CON0_merge;
#endif
#if TK_LP_dataprocess_mode == Poll
    TKS->CON0 = TK_reg_CON0_poll;
#endif
    TKS->CON1 = TK_reg_CON1_sleep;
    TKS->CHEN = TK_reg_TKSEL_LP;
    TKS->GAIN = TK_Amplifi_setting;
    TKS->SFJTR = TK_reg_SFJTR;
    TKS->IE = 0x00000010;
    TKS->IFC = 0x00000017;
}

/**
  * @brief  Initializate PIS.
  * @retval None
  */
void PIS_init(void)
{
    PIS->CH_CON[2] = 0x00001701;
}

/**
  * @brief  Initializate LPtimer.
  * @retval None
  */
void LPTIM_init(void)
{
    lptim_trigger_init_t trigger_config;
    lptim_clock_source_init_t clock_config;

    memset(&h_lptim, 0x0, sizeof(h_lptim));
    memset(&trigger_config, 0x0, sizeof(trigger_config));
    memset(&clock_config, 0x0, sizeof(clock_config));

    ald_cmu_perh_clock_config(CMU_PERH_LPTIM0, ENABLE);

    /* Initialize LPTIM0 */
    h_lptim.perh       = LPTIM0;
    h_lptim.init.psc   = LPTIM_PRESC_1;           // can not select other premeter
    h_lptim.init.arr   = TK_LPtimer_sleepsetting;
    h_lptim.init.clock = CMU_LP_PERH_CLOCK_SEL_ULRC;        //10KHz
    h_lptim.init.mode  = LPTIM_MODE_CONTINUOUS;
    ald_lptim_toggle_init(&h_lptim);

    /* Initialize clock source */
    clock_config.sel      = LPTIM_CKSEL_INTERNAL;
    clock_config.polarity = LPTIM_CKPOL_RISING;
    ald_lptim_clock_source_config(&h_lptim, &clock_config);

    /* Initialize toggle */
    trigger_config.mode = LPTIM_TRIGEN_SW;
    ald_lptim_trigger_config(&h_lptim, &trigger_config);

    /* Start LPTIM0 */
    ald_lptim_toggle_start(&h_lptim);
}

/**
  * @brief  TK sleep process.
  * @retval None
  */
void TKStandby(void)
{
    uint8_t i;
    uint16_t cmp1 = 0;
    uint16_t cmp2 = 0;

    /* Enter stop2 mode */
    ald_pmu_stop2_enter();

    /* Stop LPTIM0 */
    ald_lptim_toggle_stop(&h_lptim);

    if (g_tk_done_if)
    {
        g_tk_done_if = 0;
#if TK_LP_dataprocess_mode  == Merge

        if (TK_Sleep_cnt < 250)
        {
            cmp1 = (uint16_t)TKS->SCANRES;
#if TK_UARToutput_function == ON
            TK_md2value_origin = cmp1;
#endif

            if (cmp1 != 0)
            {
                cmp2 = (uint16_t)(TK_md2average_backup + TK_Sleep_Threshold);

                if (cmp1 > cmp2)
                {
                    if (TK_md2average_backup == 0)
                    {
                        TK_md2average_backup = cmp1;
//ս˯״̬ʱаӦѣֵް״̬ʱֵޡ
//                      if(cmp1 > )
//                      {
//                          Customer_wakeupfromsleepmode_init();
//                      }
                    }
                    else
                    {
                        Customer_wakeupfromsleepmode_init();
                    }
                }
                else
                {
                    if (TK_Sleep_cnt % 10 == 0)
                    {
                        if (cmp1 < TK_md2average_backup)
                        {
                            TK_md2average_backup--;
                        }

                        if (cmp1 > TK_md2average_backup)
                        {
                            //TK_md2average_backup++;
                            TK_md2average_backup = cmp1;
                        }
                    }
                }
            }

            if (TK_Sleep_cnt == 249)
            {
                TKS->CON0 |= 0x00000010;
            }
        }
        else//˯ɨ»
        {
            for (i = 0; i < TK_NUM; i++)
            {
                TK_value = (uint16_t)TKS->CHRES[Channel_table[i]];//TK_sleep_temp[i];
#if  Max_Minvalueoff_filter == ON
                TK_value *= TK_Samples_perscan - 2;
#else
                TK_value *= TK_Samples_perscan;
#endif
                TK_Value_Arr[i].tk_value_filter = DataFilter(TK_value, TK_Value_Arr[i].tk_value_filter);
                TK_Value_Arr[i].tk_value_average = DataFilter(TK_Value_Arr[i].tk_value_filter, TK_Value_Arr[i].tk_value_average);
            }

            TK_Sleep_cnt = 0;
            TKS->CON0 &= ~0x00000010;
        }

#endif
#if TK_LP_dataprocess_mode == Poll

        if (TK_Sleep_cnt < 250)
        {
            for (i = 0; i < TK_NUM_LP; i++)
            {
                cmp1 = (uint16_t)TKS->CHRES[Channel_table_LP[i]];
#if TK_UARToutput_function == ON
                TK_md2value_origin[i] = cmp1;
#endif

                if (cmp1 != 0)
                {
                    cmp2 = (uint16_t)(TK_md2average_backup[i] + Threshold_table_LP[i]);

                    if (cmp1 > cmp2)
                    {
                        if (TK_md2average_backup[i] == 0)
                        {
                            TK_md2average_backup[i] = cmp1;
//ս˯״̬ʱаӦѣֵް״̬ʱֵޡ
//                      if(cmp1 > )
//                      {
//                          Customer_wakeupfromsleepmode_init();
//                          break;
//                      }
                        }
                        else
                        {
//                        TK_state |= 1 << i;
//                        TK_Value_Arr[i].tk_value_filter = TK_Value_Arr[i].tk_value_average + Threshold_table[i] + 10;
//                        TK_Value_Arr[i].tk_value_origin += Threshold_table[i] + 10;
//                        TK_Value_Arr[i].tk_press_table = TK_Debounce_press + 2;
//                        TK_Value_Arr[i].tk_release_table = 0;
                            Customer_wakeupfromsleepmode_init();
                            break;
                        }
                    }
                    else
                    {
                        if (TK_Sleep_cnt % 10 == 0)
                        {
                            if (cmp1 < TK_md2average_backup[i])
                            {
                                TK_md2average_backup[i]--;
                            }

                            if (cmp1 > TK_md2average_backup[i])
                            {
                                //TK_md2average_backup[i]++;
                                TK_md2average_backup[i] = cmp1;
                            }
                        }
                    }
                }
            }

            if (TK_Sleep_cnt == 249)
            {
                TKS->CHEN = TK_Channel_Sel;
            }

#if TK_UARToutput_function == ON
#if TK_UART_TYPE == TKM_UART
            UART_Init();
#endif
#if TK_UART_TYPE == TKM_USART
            USART_Init();
#endif
#if TK_UART_TYPE == TKM_LPUART
            LPUART_Init();
#endif
            uart_send_flag = 1;
#endif
        }
        else//˯ɨ»
        {
            for (i = 0; i < TK_NUM; i++)
            {
                TK_value = (uint16_t)TKS->CHRES[Channel_table[i]];//TK_sleep_temp[i];
#if  Max_Minvalueoff_filter == ON
                TK_value *= TK_Samples_perscan - 2;
#else
                TK_value *= TK_Samples_perscan;
#endif
                TK_Value_Arr[i].tk_value_filter = DataFilter(TK_value, TK_Value_Arr[i].tk_value_filter);
                TK_Value_Arr[i].tk_value_average = DataFilter(TK_Value_Arr[i].tk_value_filter, TK_Value_Arr[i].tk_value_average);
            }

            TK_Sleep_cnt = 0;
            TKS->CHEN = TK_reg_TKSEL_LP;
        }

#endif

#if TK_UARToutput_function == ON

        if (g_wakeup_flag == 0)
        {
            if (uart_send_flag)
            {
                while (uart_send_flag != 4)
                {
                }
            }

#if TK_UART_TYPE == TKM_UART
            ald_mcu_irq_config(UART0_IRQn, 1, DISABLE);
            ald_uart_reset(&h_uart);
            ald_cmu_perh_clock_config(CMU_PERH_UART0, DISABLE);
#endif
#if TK_UART_TYPE == TKM_USART
            ald_mcu_irq_config(USART0_IRQn, 1, DISABLE);
            ald_usart_reset(&h_usart0);
            ald_cmu_perh_clock_config(CMU_PERH_USART0, DISABLE);
#endif
#if TK_UART_TYPE == TKM_LPUART
            ald_mcu_irq_config(LPUART0_IRQn, 1, DISABLE);
            ald_lpuart_reset(&h_lpuart);
            ald_cmu_perh_clock_config(CMU_PERH_LPUART0, DISABLE);
#endif

        }

        uart_send_flag = 0;
#endif

        if (g_wakeup_flag == 0)
        {
            TK_Sleep_cnt++;
            /* Start LPTIM0 */
            ald_lptim_toggle_start(&h_lptim);
        }
    }
}

#endif
