/*
 * Copyright (C) 2018 Shanghai Eastsoft Microelectronics Co., Ltd.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Change Logs:
 * Date           Author        Notes
 * 2019-01-23     wangyq        the first version
 */

#include <string.h>
#include <rthw.h>
#include <rtthread.h>
#include "board.h"
#include "drv_uart.h"
#include "drv_gpio.h"
#include <ald_cmu.h>
#include <ald_gpio.h>
#include "tkm_config.h"
#include "tk_handle.h"

/**
 * @addtogroup es32h0
 */

/*@{*/

/*******************************************************************************
* Function Name  : NVIC_Configuration
* Description    : Configures Vector Table base location.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Configuration(void)
{
}

/*******************************************************************************
 * Function Name  : SystemClock_Configuration
 * Description    : Configures the System Clock.
 * Input          : None
 * Output         : None
 * Return         : None
 *******************************************************************************/
void  SystemClock_Config(void)
{
    /* hosc 12MHz, from hosc/3 pll to 48MHz */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);

    /*  MCLK 48MHz*/
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
}

/*******************************************************************************
 * Function Name  : SysTick_Configuration
 * Description    : Configures the SysTick for OS tick.
 * Input          : None
 * Output         : None
 * Return         : None
 *******************************************************************************/
void  SysTick_Configuration(void)
{
    /* ticks = sysclk / RT_TICK_PER_SECOND */
    SysTick_Config(ald_cmu_get_sys_clock() / RT_TICK_PER_SECOND);
}

/**
 * This is the timer interrupt service routine.
 *
 */
void SysTick_Handler(void)
{
    /* enter interrupt */
    rt_interrupt_enter();
    ald_inc_tick();
    rt_tick_increase();

    /* leave interrupt */
    rt_interrupt_leave();
}

/**
  * @brief  Initializate TK moudle.
  * @retval None
  */
void TK_Init(void)
{
    TKValueStru *ptr;

    Tkscan_numth        = 0;
    Tkscan_sampcounter  = 0;
    Opr_state           = 0x8000;//Enable scan
    TK_state            = 0;
    TK_Base_percounter  = 0;
    TK_value_get        = 0;
//    Timer_10ms          = 0;
    Lock_averageconter  = 0;
    Timer_counter = 0;

    for (ptr = TK_Value_Arr; ptr < TK_Value_Arr + TK_NUM; ptr++) //Clear all
    {
        memset(ptr, 0, sizeof(TKValueStru));
    }

    /* Enable the TK clock */
    ald_cmu_tks_clock_select(CMU_TKS_SEL_HRC);
    ald_cmu_perh_clock_config(CMU_PERH_TKS, ENABLE);
    ald_mcu_irq_config(TKS_IRQn, 1, ENABLE);

    TKS->CHEN = TK_Channel_Sel;
    TKS->CON0 = TK_reg_CON0;        //24Ƶ  1/2ռձ  ΪŴϵԳŵֵ
    TKS->CON1 = TK_reg_CON1;        //CpCx  Cxŵ256ʱ  2.5V
    TKS->GAIN = 0x000FFFFF;
    TKS->SFJTR = TK_reg_SFJTR;      //΢Ƶ
    TKS->IE = 0x10;                 //ʹж
    TKS->IFC = 0x1F;                //жϱ־
}

/*@}*/
/**
 * This function will initial ES32F0 board.
 */
void rt_hw_board_init(void)
{
    /* NVIC Configuration */
    NVIC_Configuration();

    /*System Clock Configuration */
    SystemClock_Config();

    /* Configure the SysTick */
    SysTick_Configuration();

    ald_cmu_perh_clock_config(CMU_PERH_UART0, ENABLE);

    rt_hw_uart_init();

#ifdef RT_USING_HEAP
    rt_system_heap_init((void *)HEAP_BEGIN, (void *)HEAP_END);
#endif
#ifdef RT_USING_COMPONENTS_INIT
    rt_components_board_init();
#endif
#ifdef RT_USING_CONSOLE
    rt_console_set_device(RT_CONSOLE_DEVICE_NAME);
#endif

    TK_Init();
}

/**
 * This function will delay for some us.
 *
 * @param us the delay time of us
 */
void rt_hw_us_delay(rt_uint32_t us)
{
    unsigned int start, now, delta, reload, us_tick;
    start = SysTick->VAL;
    reload = SysTick->LOAD;
    us_tick = ald_cmu_get_sys_clock() / 1000000UL;

    do
    {
        now = SysTick->VAL;
        delta = start > now ? start - now : reload + start - now;
    }
    while (delta <  us_tick * us);
}
