/**
  *********************************************************************************
  *
  * @file    timer.c
  * @brief   timer moudle for DEMO
  *
  * @version V1.0
  * @date    26 Apr 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Apr 2021     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *********************************************************************************
  */

#include <string.h>
#include "ald_cmu.h"
#include "ald_gpio.h"
#include "ald_timer.h"
#include "timer.h"

timer_handle_t h_tim;
timer_clock_config_t clock;
timer_ic_init_t init;

/* Captured Values */
uint32_t value1 = 0;
uint32_t value2 = 0;
uint32_t diff = 0;
uint16_t idx = 0;
uint32_t freq = 0;

uint32_t data_result[50] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
uint8_t full_data = 0;
uint8_t cnt_num = 0;

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_init(void)
{
    gpio_init_t x;
    memset(&x, 0x0, sizeof(x));

    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);

    /* Initialize GPIO_PIN_7 sa output */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.func = GPIO_FUNC_1;
    ald_gpio_init(GPIOC, GPIO_PIN_7, &x);   //test pin
}

/**
  * @brief  Input capture callback  function
  * @retval None
  */
void ald_timer_ic_capturecallback(struct timer_handle_s *htim)
{
    if (full_data == 0)
    {
        ald_gpio_toggle_pin(GPIOC, GPIO_PIN_7); //test pin

        if (idx == 0)
        {
            value1 = GP16C4T0->CCVAL1;
            idx = 1;
        }
        else
        {
            value2 = GP16C4T0->CCVAL1;
            idx = 0;

            if (value2 > value1)
            {
                diff = (value2 - value1);
            }
            else if (value2 < value1)
            {
                /* 0xFFFF is max TIM0_CCRx value */
                diff = ((0xFFFF - value1) + value2) + 1;
            }

            data_result[cnt_num] = diff;
            cnt_num++;

            if (cnt_num > 49)
            {
                cnt_num = 0;
                full_data = 1;
            }
        }
    }

    return;
}

void Timer_Init(void)
{
    /* Initialize pin and set the default state*/
    timer_pin_init();//for test

    ald_cmu_perh_clock_config(CMU_PERH_GP16C4T0, ENABLE);

    /* Initialize GP16C4T0 */
    h_tim.perh           = GP16C4T0;
    h_tim.init.prescaler = 0;
    h_tim.init.mode      = TIMER_CNT_MODE_UP;
    h_tim.init.period    = 0xFFFF;
    h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
    h_tim.init.re_cnt    = 0;
    h_tim.capture_cbk    = ald_timer_ic_capturecallback;
    ald_timer_ic_init(&h_tim);

    /* Initialize clock source */
    clock.source     = TIMER_SRC_INTER;
    ald_timer_config_clock_source(&h_tim, &clock);

    /* Initialize input capture channel */
    init.polarity = TIMER_IC_POLARITY_RISE;
    init.sel      = TIMER_IC_SEL_DIRECT;
    init.psc      = TIMER_IC_PSC_DIV1;
    init.filter   = 0;
    ald_timer_ic_config_channel(&h_tim, &init, TIMER_CHANNEL_1);
    ald_mcu_irq_config(GP16C4T0_LCD_IRQn, 1, ENABLE);

    /* Enable CC interrupt */
    ald_timer_interrupt_config(&h_tim, TIMER_IT_CC1, ENABLE);

    /* Start input capture */
    ald_timer_ic_start(&h_tim, TIMER_CHANNEL_1);
    ald_timer_ic_start_by_it(&h_tim, TIMER_CHANNEL_1);
}
