/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uint32_t __cnt = 0;
uart_handle_t h_uart;
char uart_buf[64];

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_96M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);
	/* Enable peripheral clock */
	ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
	ald_cmu_perh_clock_config(CMU_PERH_UART0, ENABLE);

	memset(uart_buf, 0x0, 64);
	uart_stdio_init();
	light_init();

	printf_e("System start...\n\r");
	light_ctrl(LIGHT_IDX_2, LIGHT_CTRL_ON);
	ald_delay_ms(2000);
	light_ctrl(LIGHT_IDX_2, LIGHT_CTRL_OFF);
	ald_delay_ms(1000);

	while (1) {
		light_ctrl(LIGHT_IDX_1, LIGHT_CTRL_TOGGLE);
		light_ctrl(LIGHT_IDX_2, LIGHT_CTRL_TOGGLE);
		printf_e("Count: %d\n\r", __cnt++);
		ald_delay_ms(200);
	}
}

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(uart_buf, 64, fmt, args);
	va_end(args);

	ald_uart_send(&h_uart, (uint8_t *)uart_buf, strlen(uart_buf), 1000);
	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = UART_FUNC_TX;
	ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

	/* Initialize rx pin */
	x.mode  = GPIO_MODE_INPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	x.func  = UART_FUNC_RX;
	ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);

	return;
}

/**
  * @brief  Initialize light.
  * @retval None
  */
void light_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.func = GPIO_FUNC_1;

	ald_gpio_init(LED1_PORT, LED1_PIN, &x);
	ald_gpio_init(LED2_PORT, LED2_PIN, &x);
	light_ctrl(LIGHT_IDX_1, LIGHT_CTRL_OFF);
	light_ctrl(LIGHT_IDX_2, LIGHT_CTRL_OFF);

	return;
}

/**
  * @brief  Control light.
  * @param  idx: Index of the light.
  * @param  ctrl: Type: ON/OFF/TOGGLE.
  * @retval None
  */
void light_ctrl(light_idx_t idx, light_ctrl_t ctrl)
{
	if (idx == LIGHT_IDX_1) {
		switch ((uint32_t)ctrl) {
		case LIGHT_CTRL_ON:
			ald_gpio_write_pin(LED1_PORT, LED1_PIN, 0);
			break;
		case LIGHT_CTRL_OFF:
			ald_gpio_write_pin(LED1_PORT, LED1_PIN, 1);
			break;
		case LIGHT_CTRL_TOGGLE:
			ald_gpio_toggle_pin(LED1_PORT, LED1_PIN);
			break;
		default:
			break;
		}
	}
	else if (idx == LIGHT_IDX_2) {
		switch ((uint32_t)ctrl) {
		case LIGHT_CTRL_ON:
			ald_gpio_write_pin(LED2_PORT, LED2_PIN, 0);
			break;
		case LIGHT_CTRL_OFF:
			ald_gpio_write_pin(LED2_PORT, LED2_PIN, 1);
			break;
		case LIGHT_CTRL_TOGGLE:
			ald_gpio_toggle_pin(LED2_PORT, LED2_PIN);
			break;
		default:
			break;
		}
	}
	else {
		return;
	}

	return;	
}
/**
  * @}
  */
/**
  * @}
  */
