/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

md_dma_descriptor_t dma0_ctrl_base[28] __attribute__ ((aligned(512)));
md_dma_config_t config1;
md_dma_config_t config2;

volatile uint8_t __flag = 0;
uint32_t src1_buf[PWM_SIZE];
uint32_t src2_buf[PWM_SIZE];

/** @addtogroup Examples
  * @{
  */
  
/**
  * @brief  Initialize pwm.
  * @param  None
  * @retval None
  */
void PWM_gpio_init(void)
{
	md_gpio_set_pin_function(TIMER_GPIO_PORT, TIMER_GPIO_PIN, MD_GPIO_FUNC_2);
	md_gpio_set_pin_push_pull(TIMER_GPIO_PORT, TIMER_GPIO_PIN);
	md_gpio_set_pin_mode_output(TIMER_GPIO_PORT, TIMER_GPIO_PIN);
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void timer_base_init(void)
{
	md_timer_base_init_t init;

	md_timer_base_struct_init(&init);
	init.period = PERIOD_VALUE;
	md_timer_base_set_config(GP32C4T0, &init);
	
	return;
}

/**
  * @brief  Initialize TIMER_oc.
  * @param  None
  * @retval None
  */
void timer_oc_init(void)
{
	md_timer_oc_init_t init;
	
	init.ocstate	  = MD_TIMER_OCSTATE_DISABLE;
	init.ocnstate     = MD_TIMER_OCSTATE_DISABLE;
	init.oc_mode      = MD_TIMER_OC_MODE_PWM1;
	init.pulse        = 5;
	init.oc_polarity  = MD_TIMER_OC_POLARITY_HIGH;
	init.ocn_polarity = MD_TIMER_OCN_POLARITY_HIGH;
	init.oc_idle      = MD_TIMER_OC_IDLE_RESET;
	init.ocn_idle     = MD_TIMER_OCN_IDLE_RESET;
	init.oc_fast_en   = DISABLE;
	
	md_timer_oc_init(GP32C4T0, MD_TIMER_CHANNEL_1, &init);
	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{	
	uint32_t i;
	
	/* Configure system clock */
	md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
	md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

	/* Enable ALL peripheral */
	SYSCFG_UNLOCK();
	md_cmu_enable_perh_all();
	SYSCFG_LOCK();

	/* Initialize pin */
	PWM_gpio_init();
	/* Initialize timer */
	timer_base_init();
	/* Initialize timer_oc */
	timer_oc_init();
	
	/* Initialize dma */
	md_dma_reset(DMA0);
	NVIC_SetPriority(DMA_IRQn, 2);
	NVIC_EnableIRQ(DMA_IRQn);
	SET_BIT(DMA0->IER, DMA_IER_DMAERRIE_MSK);
	
	memset(dma0_ctrl_base, 0x0, sizeof(dma0_ctrl_base));
	md_dma_set_ctrlbase((uint32_t)&dma0_ctrl_base);
	md_dma_enable();
	
	for(i=0; i<PWM_SIZE; i++) {
		src1_buf[i] = PWM_TEN;
		src2_buf[i] = PWM_HALF;
	}
	
	/* Initialize config1 */
	memset(&config1, 0x0, sizeof(md_dma_config_t));
	config1.src        = src1_buf;
	config1.dst        = (void *)&GP32C4T0->CCVAL1;
	config1.size       = PWM_SIZE;
	config1.data_width = MD_DMA_DATA_SIZE_WORD;
	config1.src_inc    = MD_DMA_DATA_INC_WORD;
	config1.dst_inc    = MD_DMA_DATA_INC_NONE;
	config1.R_power    = MD_DMA_R_POWER_1;
	config1.primary    = ENABLE;
	config1.burst      = ENABLE;
	config1.high_prio  = DISABLE;
	config1.interrupt  = ENABLE;
	config1.channel    = 0;
	config1.msel       = MD_DMA_MSEL_TIMER2;
	config1.msigsel    = MD_DMA_MSIGSEL_TIMER_UPDATE;
	md_dma_config_ping_pong(DMA0, &config1, 1);
	
	md_timer_enable_udma(GP32C4T0);
	/* Start input pwm from tim0 channel 1 */
	md_timer_set_cc1_output_enable_cc1en(GP32C4T0, 1);
	/* TIMER Counter enable */
	md_timer_enable_counter_cnten(GP32C4T0);

	memset(&config2, 0x0, sizeof(md_dma_config_t));	
	config2.src        = src2_buf;
	config2.dst        = (void *)&GP32C4T0->CCVAL1;
	config2.size       = PWM_SIZE;
	config2.R_power    = MD_DMA_R_POWER_1;
	config2.data_width = MD_DMA_DATA_SIZE_WORD;
	config2.src_inc    = MD_DMA_DATA_INC_WORD;;
	config2.dst_inc    = MD_DMA_DATA_INC_NONE;
	config2.primary    = DISABLE;
	config2.channel    = 0;
	md_dma_config_ping_pong(DMA0, &config2, 0);
	
	/* Wait for the first transfer to complete */
	while (!__flag);
	__flag = 0;
	
	while (1) {	
		memset(&config1, 0x0, sizeof(md_dma_config_t));	
		config1.src        = src1_buf;
		config1.dst        = (void *)&GP32C4T0->CCVAL1;
		config1.size       = PWM_SIZE;
		config1.R_power    = MD_DMA_R_POWER_1;
		config1.data_width = MD_DMA_DATA_SIZE_WORD;
		config1.src_inc    = MD_DMA_DATA_INC_WORD;;
		config1.dst_inc    = MD_DMA_DATA_INC_NONE;
		config1.primary    = ENABLE;
		config1.channel    = 0;
		md_dma_config_ping_pong(DMA0, &config1, 0);

		/* Wait for the transfer to complete */
		while (!__flag);
		__flag = 0;

		memset(&config2, 0x0, sizeof(md_dma_config_t));	
		config2.src        = src2_buf;
		config2.dst        = (void *)&GP32C4T0->CCVAL1;
		config2.size       = PWM_SIZE;
		config2.R_power    = MD_DMA_R_POWER_1;
		config2.data_width = MD_DMA_DATA_SIZE_WORD;
		config2.src_inc    = MD_DMA_DATA_INC_WORD;;
		config2.dst_inc    = MD_DMA_DATA_INC_NONE;
		config2.primary    = DISABLE;
		config2.channel    = 0;
		md_dma_config_ping_pong(DMA0, &config2, 0);
		
		/* Wait for the first transfer to complete */
		while (!__flag);
		__flag = 0;

	}
}

/**
  * @}
  */
/**
  * @}
  */
