/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Public_Function
  * @{
  */
md_crypt_init_t h_crypt;

/* des ecb key = 1 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6
 * refer result	61 cb 4a 69 bf cc 9b c4
 */
void test_des_ecb_1key()
{
	uint32_t plain[2]  = {0x3243F6A8, 0x885A308D};
	uint32_t result[2] = {0};
	uint32_t expect[2] = {0x61cb4a69,0xbfcc9bc4};
	uint32_t key[4]    = {0x2b7e1516, 0x28aed2a6};
	uint32_t encrypt_pos  = 0;
	uint32_t result_pos = 0;
	uint32_t decrypt_pos = 0;
	uint32_t loop = 0;
	uint32_t step = 0;
	uint32_t count = 0;

	memset(&h_crypt, 0x0, sizeof(md_crypt_init_t));

	/* DES-ECB mode step = 2 */
	step = 2;

	/* Initialize crypt */
	md_crypt_reset();
	md_crypt_init_struct(&h_crypt);
	md_crypt_init(&h_crypt);

	/* Set fifo order MD_CRYPT_FIFO_ORDER_3 */
	md_crypt_set_fifo_order(CRYPT, MD_CRYPT_FIFO_ORDER_3);
	/* Set des standard */
	md_crypt_set_standard(CRYPT, MD_CRYPT_STANDARD_DES);
	/* Set crypt cbc mode */
	md_crypt_set_mode(CRYPT, MD_CRYPT_MODE_ECB);
	/* Enable fifo */
	md_crypt_enable_fifo(CRYPT);

	/* Set key */
	md_crypt_set_key1(CRYPT, key[0]);
	md_crypt_set_key0(CRYPT, key[1]);
	/* Set ENCS bit */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_ENCRYPT);

	/* Calculate loop tomes */
	count = sizeof(plain) / (4 * step);

	/* right result value equal to expect value */
	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, plain[encrypt_pos]);
			encrypt_pos++;
		}

		while (md_crypt_get_flag_done(CRYPT) == SET);

		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}

	/* Set decrypt function */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_DECRYPT);

	count = sizeof(plain) / (4 * step);

	result_pos = 0;
	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, expect[decrypt_pos]);
			decrypt_pos++;
		}
		
		while (md_crypt_get_flag_done(CRYPT) == SET);
		
		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}
	UNUSED(result);
}

/* des ecb key = 2 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c
 * refer result	xx
 */
void test_des_ecb_2key()
{
	uint32_t plain[2]  = {0x3243F6A8, 0x885A308D};
	uint32_t result[2] = {0};
	uint32_t expect[2] = {0x1f2bca17,0x7bc9d663};
	uint32_t key[4]    = {0x2b7e1516, 0x28aed2a6, 0xabf71588, 0x09cf3c2b};
	uint32_t encrypt_pos  = 0;
	uint32_t result_pos = 0;
	uint32_t decrypt_pos = 0;
	uint32_t loop = 0;
	uint32_t step = 0;
	uint32_t count = 0;

	UNUSED(result);
	memset(&h_crypt, 0x0, sizeof(md_crypt_init_t));

	/* TDES-ECB mode step = 2 */
	step = 2;

	/* Initialize crypt */
	md_crypt_reset();
	md_crypt_init_struct(&h_crypt);
	md_crypt_init(&h_crypt);

	/* Set fifo order MD_CRYPT_FIFO_ORDER_3 */
	md_crypt_set_fifo_order(CRYPT, MD_CRYPT_FIFO_ORDER_3);
	/* Set des standard */
	md_crypt_set_standard(CRYPT, MD_CRYPT_STANDARD_DES);
	/* Enable TDES */
	md_crypt_enable_tdes(CRYPT);
	/* Set crypt cbc mode */
	md_crypt_set_mode(CRYPT, MD_CRYPT_MODE_ECB);
	/* Enable fifo */
	md_crypt_enable_fifo(CRYPT);

	/* Set key */
	md_crypt_set_key3(CRYPT, key[0]);
	md_crypt_set_key2(CRYPT, key[1]);
	md_crypt_set_key1(CRYPT, key[2]);
	md_crypt_set_key0(CRYPT, key[3]);
	/* Set ENCS bit */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_ENCRYPT);

	/* Calculate loop tomes */
	count = sizeof(plain) / (4 * step);

	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, plain[encrypt_pos]);
			encrypt_pos++;
		}

		while (md_crypt_get_flag_done(CRYPT) == SET);
		
		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}

	/* Set decrypt function */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_DECRYPT);
	count = sizeof(plain) / (4 * step);
	result_pos = 0;

	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, expect[decrypt_pos]);
			decrypt_pos++;
		}

		while (md_crypt_get_flag_done(CRYPT) == SET);

		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}
}

/* des ecb key = 3 keys
 * input 	32 43 f6 a8 88 5a 30 8d
 * cipher key	2b 7e 15 16 28 ae d2 a6 ab f7 15 88 09 cf 4f 3c 2b 7e 15 16 28 ae d2 a6
 * refer result	bd 67 f6 a6 83 df b1 52
 */
void test_des_ecb_3key()
{
	uint32_t plain[2]  = {0x3243F6A8, 0x885A308D};
	uint32_t result[2] = {0};
	uint32_t expect[2] = {0xbd67f6a6, 0x83dfb152};
	uint32_t key[6]    = {0x2b7e1516, 0x28aed2a6, 0xabf71588, 0x09cf3c2b, 0x2b7e1516, 0x28aed2a6};
	uint32_t encrypt_pos = 0;
	uint32_t result_pos = 0;
	uint32_t decrypt_pos = 0;
	uint32_t loop = 0;
	uint32_t step = 0;
	uint32_t count = 0;

	UNUSED(result);
	memset(&h_crypt, 0x0, sizeof(md_crypt_init_t));

	/* TDES-ECB 2 kye mode step = 2 */
	step = 2;

	/* Initialize crypt */
	md_crypt_reset();
	md_crypt_init_struct(&h_crypt);
	md_crypt_init(&h_crypt);

	/* Set fifo order MD_CRYPT_FIFO_ORDER_3 */
	md_crypt_set_fifo_order(CRYPT, MD_CRYPT_FIFO_ORDER_3);
	/* Set des standard */
	md_crypt_set_standard(CRYPT, MD_CRYPT_STANDARD_DES);
	/* Enable TDES */
	md_crypt_enable_tdes(CRYPT);
	/* Set teds 3 keys */
	md_crypt_set_tdes_keys(CRYPT, MD_CRYPT_TDES_KEY_3);
	/* Set crypt cbc mode */
	md_crypt_set_mode(CRYPT, MD_CRYPT_MODE_ECB);
	/* Enable fifo */
	md_crypt_enable_fifo(CRYPT);

	/* Set key */
	md_crypt_set_key5(CRYPT, key[0]);
	md_crypt_set_key4(CRYPT, key[1]);
	md_crypt_set_key3(CRYPT, key[2]);
	md_crypt_set_key2(CRYPT, key[3]);
	md_crypt_set_key1(CRYPT, key[4]);
	md_crypt_set_key0(CRYPT, key[5]);
	/* Set ENCS bit */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_ENCRYPT);

	/* Calculate loop tomes */
	count = sizeof(plain) / (4 * step);

	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, plain[encrypt_pos]);
			encrypt_pos++;
		}

		while (md_crypt_get_flag_done(CRYPT) == SET);
		
		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}

	/* Set decrypt function */
	md_crypt_set_encs(CRYPT, MD_CRYPT_SEL_DECRYPT);
	count = sizeof(plain) / (4 * step);
	result_pos = 0;

	while (count--) {
		/* Write data need encrypt to fifo */
		for (loop = 0; loop < step; loop++) {
			md_crypt_set_fifo_data(CRYPT, expect[decrypt_pos]);
			decrypt_pos++;
		}

		while (md_crypt_get_flag_done(CRYPT) == SET);

		/* Get encrypt result */
		for (loop = 0; loop < step; loop++) {
			result[result_pos] = md_crypt_get_fifo_data(CRYPT);
			result_pos++;
		}
	}
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Configure system clock */
	md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
	md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
	/* PriorityGroup_2 */
	NVIC_SetPriorityGrouping(5); 
	/* Initialize SysTick Interrupt */
	md_init_1ms_tick();

	/* Enable ALL peripheral */
	SYSCFG_UNLOCK();
	md_cmu_enable_perh_all();
	SYSCFG_LOCK();

	/* Des ecb mode test */
	test_des_ecb_1key();
	test_des_ecb_2key();
	test_des_ecb_3key();

	while (1) {
		md_delay_1ms(1000);
	}
}
/**
  * @}
  */
/**
  * @}
  */
