/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

uart_handle_t h_uart;
uint8_t tx_buf[32];
uint8_t rx_buf[32];
volatile uint8_t tx_flag;
volatile uint8_t rx_flag;
volatile uint32_t tx_cnt;
volatile uint32_t rx_cnt;
extern dma_descriptor_t dma0_ctrl_base[28];

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void uart_pin_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x);

	return;
}

/**
  * @brief  Configure DMA for UART module.
  * @retval None
  */
void usr_uart_dma_init(void)
{
	/* configure TX */
	h_uart.hdmatx.perh     = DMA0;
	h_uart.hdmatx.cplt_cbk = NULL;
	h_uart.hdmatx.cplt_arg = NULL;
	h_uart.hdmatx.err_cbk  = NULL;
	h_uart.hdmatx.err_arg  = NULL;

	ald_dma_config_struct(&h_uart.hdmatx.config);
	h_uart.hdmatx.config.src     = (void *)0x20000000;
	h_uart.hdmatx.config.dst     = (void *)&h_uart.perh->TXBUF;
	h_uart.hdmatx.config.size    = 0;
	h_uart.hdmatx.config.src_inc = DMA_DATA_INC_BYTE;
	h_uart.hdmatx.config.dst_inc = DMA_DATA_INC_NONE;
	h_uart.hdmatx.config.msigsel = DMA_MSIGSEL_UART_TXEMPTY;
	h_uart.hdmatx.config.msel    = DMA_MSEL_UART0;
	h_uart.hdmatx.config.burst   = ENABLE;
	h_uart.hdmatx.config.channel = UART_TX_DMA_CH;
	ald_dma_config_basic(&h_uart.hdmatx);
	DMA0->CHENCLR = (1 << UART_TX_DMA_CH);

	/* configure RX */
	h_uart.hdmarx.perh     = DMA0;
	h_uart.hdmarx.cplt_cbk = NULL;
	h_uart.hdmarx.cplt_arg = NULL;
	h_uart.hdmarx.err_cbk  = NULL;
	h_uart.hdmarx.err_arg  = NULL;

	ald_dma_config_struct(&h_uart.hdmarx.config);
	h_uart.hdmarx.config.src     = (void *)&h_uart.perh->RXBUF;
	h_uart.hdmarx.config.dst     = (void *)0x20000000;
	h_uart.hdmarx.config.size    = 0;
	h_uart.hdmarx.config.src_inc = DMA_DATA_INC_NONE;
	h_uart.hdmarx.config.dst_inc = DMA_DATA_INC_BYTE;
	h_uart.hdmarx.config.msigsel = DMA_MSIGSEL_UART_RNR;
	h_uart.hdmarx.config.msel    = DMA_MSEL_UART0;
	h_uart.hdmarx.config.burst   = ENABLE;
	h_uart.hdmarx.config.channel = UART_RX_DMA_CH;
	ald_dma_config_basic(&h_uart.hdmarx);
	DMA0->CHENCLR = (1 << UART_RX_DMA_CH);

	return;
}

/**
  * @brief  Start TX by DMA.
  * @param  buf: Pointer to the buffer
  * @param  size: Size of the buffer
  * @retval None
  */
void usr_uart_tx_start(uint8_t *buf, uint32_t size)
{
	tx_flag = 0;
	SET_BIT(h_uart.perh->FCON, UART_FCON_TFRST_MSK);
	h_uart.perh->TXBUF = buf[0];

	if (size == 1) {
		tx_flag = 1;
		return;
	}

	dma0_ctrl_base[UART_TX_DMA_CH].src = (void *)(buf + size - 1);
	dma0_ctrl_base[UART_TX_DMA_CH].ctrl.n_minus_1  = size - 2;
	dma0_ctrl_base[UART_TX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
	DMA0->CHENSET = (1 << UART_TX_DMA_CH);
	SET_BIT(h_uart.perh->MCON, UART_MCON_TXDMAEN_MSK);

	return;
}

/**
  * @brief  Start RX by DMA.
  * @param  buf: Pointer to the buffer
  * @param  size: Size of the buffer
  * @retval None
  */
void usr_uart_rx_start(uint8_t *buf, uint32_t size)
{
	rx_flag = 0;
	SET_BIT(h_uart.perh->FCON, UART_FCON_RFRST_MSK);
	dma0_ctrl_base[UART_RX_DMA_CH].dst = (void *)(buf + size - 1);
	dma0_ctrl_base[UART_RX_DMA_CH].ctrl.n_minus_1  = size - 1;
	dma0_ctrl_base[UART_RX_DMA_CH].ctrl.cycle_ctrl = DMA_CYCLE_CTRL_BASIC;
	DMA0->CHENSET = (1 << UART_RX_DMA_CH);
	SET_BIT(h_uart.perh->MCON, UART_MCON_RXDMAEN_MSK);

	return;
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* clear uart_handle_t structure */
	memset(&h_uart, 0x0, sizeof(h_uart));

	/* Initialize pin */
	uart_pin_init();

	/* Initialize tx_buf */
	memset(tx_buf, 0x55, sizeof(tx_buf));

	/* Initialize uart */
	h_uart.perh             = UART0;
	h_uart.init.baud        = 115200;
	h_uart.init.word_length = UART_WORD_LENGTH_8B;
	h_uart.init.stop_bits   = UART_STOP_BITS_1;
	h_uart.init.parity      = UART_PARITY_NONE;
	h_uart.init.mode        = UART_MODE_UART;
	h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	h_uart.tx_cplt_cbk      = NULL;
	h_uart.rx_cplt_cbk      = NULL;
	h_uart.error_cbk        = NULL;
	ald_uart_init(&h_uart);
	usr_uart_dma_init();

	tx_cnt = 0;
	rx_cnt = 0;

	usr_uart_tx_start(rx_buf, 32);
	usr_uart_rx_start(rx_buf, 32);

	while (1) {
		if (tx_flag) {
			usr_uart_tx_start(tx_buf, 32);
			++tx_cnt;
		}

		if (rx_flag) {
			usr_uart_rx_start(rx_buf, 32);
			++rx_cnt;
		}
	}
}

/**
  * @}
  */
/**
  * @}
  */
