/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */
/** @addtogroup Examples
  * @{
  */

uart_handle_t h_uart0;
uart_handle_t h_uart5;
uint8_t tx_buf[32];
uint8_t rx_buf[32];

/**
  * @brief  Initializate pin of uart Smart Card module.
  * @retval None
  */
void uart_pin_scard_init(void)
{
	gpio_init_t x;
	memset(&x, 0x00, sizeof(gpio_init_t));

	/* Initialize UART5 TX pin: PA2 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_OPEN_DRAIN;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(CARD_IO5_PORT, CARD_IO5_PIN, &x);

	/* Initialize UART5 RST pin: PA1 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_1;
	ald_gpio_init(CARD_RST_PORT, CARD_RST_PIN, &x);

	/* Initialize UART5 CLK pin: PA4 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(CARD_CLK_PORT, CARD_CLK_PIN, &x);
	
	/* Initialize UART0 TX pin PB10 */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_FLOATING;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(CARD_IO0_PORT, CARD_IO0_PIN, &x);

	return;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  hperh: Pointer to a uart_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  len: Amount of data to be sent
  * @retval result
  */
uint8_t virtual_7816_data_recv(uart_handle_t *hperh, uint8_t *buf, uint8_t len)
{
	uint32_t i, out_time;
	uint32_t clk = ald_cmu_get_pclk1_clock();

	DISABLE_UART_TX(hperh);
	ENABLE_UART_RX(hperh);

	for (i = 0; i < len; i++) {
		for (out_time = 0; out_time < clk; out_time++) {
			if (READ_BIT(hperh->perh->STAT, UART_STAT_RFTH_MSK) == UART_STAT_RFTH_MSK)
			    break;
		}

		if (out_time == clk)
			return i + 1;
		if (READ_BIT(hperh->perh->STAT, UART_STAT_PERR_MSK) == UART_STAT_PERR_MSK)
			return i + 1;

		buf[i] = hperh->perh->RXBUF;
	}

	return 0;
}

/**
  * @brief  Transmit an amount of data in blocking mode.
  * @param  hperh: Pointer to a uart_handle_t structure.
  * @param  buf: Pointer to data buffer
  * @param  len: Amount of data to be sent
  * @retval result
  */
uint8_t virtual_7816_data_send(uart_handle_t *hperh, uint8_t *buf, uint8_t len)
{
	uint32_t i, out_time;
	uint32_t clk = ald_cmu_get_pclk1_clock();

	DISABLE_UART_RX(hperh);
	ENABLE_UART_TX(hperh);

	for (i = 0; i < len; i++) {
		hperh->perh->TXBUF = buf[i];

		for (out_time = 0; out_time < clk; out_time++) {
			if (READ_BIT(hperh->perh->RIF, UART_RIF_TBC_MSK) == UART_RIF_TBC_MSK)
				break;
		}

		ald_uart_clear_flag_status(hperh, UART_IF_TBC);

		if (out_time == clk) {
			DISABLE_UART_TX(hperh);
			ENABLE_UART_RX(hperh);

			return i + 1;
		}

		if (READ_BIT(hperh->perh->STAT, UART_STAT_PERR_MSK) == UART_STAT_PERR_MSK) {
			DISABLE_UART_TX(hperh);
			ENABLE_UART_RX(hperh);

			return i + 1;
		}
	}

	DISABLE_UART_TX(hperh);
	ENABLE_UART_RX(hperh);

	return 0;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_clock_config(CMU_CLOCK_HRC, 24000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* clear uart_handle_t structure */
	memset(&h_uart0, 0x0, sizeof(h_uart0));
	memset(&h_uart5, 0x0, sizeof(h_uart5));
	memset(tx_buf, 0x55, sizeof(tx_buf));
	memset(rx_buf, 0x0, sizeof(tx_buf));

	/* Initialize pin */
	uart_pin_scard_init();

	/* Initialize UART5 Smart Card */
	h_uart5.perh             = UART5;
	h_uart5.init.baud        = 16129;
	h_uart5.init.word_length = UART_WORD_LENGTH_8B;
	h_uart5.init.stop_bits   = UART_STOP_BITS_1_5;
	h_uart5.init.parity      = UART_PARITY_EVEN;
	h_uart5.init.mode        = UART_MODE_SCARD;
	h_uart5.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart5);

	UART_ENABLE_SCARD_CLK(&h_uart5);
	UART_ENABLE_SCARD_NACK(&h_uart5);
	MODIFY_REG(h_uart5.perh->SCARD, UART_SCARD_GT_MSK, 2 << UART_SCARD_GT_POSS);
	MODIFY_REG(h_uart5.perh->SCARD, UART_SCARD_PSC_MSK, 1 << UART_SCARD_PSC_POSS);
	UART_SCARD_ENABLE(&h_uart5);
	ald_uart_clear_flag_status(&h_uart5, UART_IF_TBC);

	/* Initialize UART0 */
	h_uart0.perh             = UART0;	
	h_uart0.init.baud        = 16129;
	h_uart0.init.word_length = UART_WORD_LENGTH_8B;
	h_uart0.init.stop_bits   = UART_STOP_BITS_1_5;
	h_uart0.init.parity      = UART_PARITY_EVEN;
	h_uart0.init.mode        = UART_MODE_HDSEL;
	h_uart0.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	ald_uart_init(&h_uart0);

	ald_uart_rx_fifo_config(&h_uart0, UART_RXFIFO_1BYTE);
	ald_uart_tx_fifo_config(&h_uart0, UART_TXFIFO_EMPTY);
	ald_uart_clear_flag_status(&h_uart0, UART_IF_TBC);	

	virtual_7816_data_send(&h_uart0, tx_buf, 10);
	virtual_7816_data_recv(&h_uart0, rx_buf, 10);

	while (1) {
		ald_delay_ms(1000);
	}
}
/**
  * @}
  */
/**
  * @}
  */
