/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include <string.h>

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

pis_handle_t h_pis;

timer_handle_t h_tim0;
timer_handle_t h_tim1;
timer_handle_t h_tim2;

timer_slave_config_t tim_slave;
timer_master_config_t tim_master;

/**
  * @brief  Initializate timer module.
  * @retval None
  */
void pis_tim_init(void)
{
	/* clear timer_handle_t structure */
	memset(&h_tim0,0,sizeof(timer_handle_t));
	memset(&h_tim1,0,sizeof(timer_handle_t));
	memset(&h_tim2,0,sizeof(timer_handle_t));
	memset(&tim_slave,0,sizeof(tim_slave));
	memset(&tim_master,0,sizeof(tim_master));
	
	/* Initialize slave timer0 */
	h_tim0.perh           = TIMER0;
	h_tim0.init.prescaler = 0x03;
	h_tim0.init.mode      = TIMER_CNT_MODE_UP;
	h_tim0.init.period    = 0x300;
	h_tim0.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim0.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim0);
	
	/* Initialize slave timer1 */
	h_tim1.perh           = TIMER1;
	h_tim1.init.prescaler = 0x03;
	h_tim1.init.mode      = TIMER_CNT_MODE_UP;
	h_tim1.init.period    = 0x300;
	h_tim1.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim1.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim1);
	
	/* Config slave receive */
	tim_slave.mode     = TIMER_MODE_TRIG;
	tim_slave.input    = TIMER_TS_ITR0;
	tim_slave.filter   = 0;
	tim_slave.polarity = TIMER_CLK_POLARITY_RISE;
	tim_slave.psc      = TIMER_ETR_PSC_DIV1;
	ald_timer_slave_config_sync(&h_tim0, &tim_slave);
	
	tim_slave.input    = TIMER_TS_ITR1;
	ald_timer_slave_config_sync(&h_tim1, &tim_slave);
	
	/* Initialize master timer2 */
	h_tim2.perh           = TIMER2;
	h_tim2.init.prescaler = 4699;
	h_tim2.init.mode      = TIMER_CNT_MODE_UP;
	h_tim2.init.period    = 65535;
	h_tim2.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim2.init.re_cnt    = 0;
	ald_timer_base_init(&h_tim2);
	
	/* config master trigger out */
	tim_master.master_en  = ENABLE;
	tim_master.sel        = TIMER_TRGO_ENABLE;
	ald_timer_master_sync_config(&h_tim2, &tim_master);
}

/**
  * @brief  Start timer2.
  * @retval None
  */
void pis_tim_start(void)
{
	ald_timer_base_start(&h_tim2);
}

/**
  * @brief  Stop timer2 module.
  * @retval None
  */
void pis_tim_stop(void)
{
	ald_timer_base_stop(&h_tim2);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Clear pis_handle_t struct */
	memset(&h_pis, 0, sizeof(pis_handle_t));

	/* Initialize timer */
	pis_tim_init();

	/* Initialize PIS */
	h_pis.perh                 = PIS;
	h_pis.init.producer_src    = PIS_TIMER2_UPDATA;
	h_pis.init.producer_clk    = PIS_CLK_PCLK1;
	h_pis.init.producer_edge   = PIS_EDGE_UP;
	h_pis.init.producer_signal = PIS_OUT_PULSE;
	h_pis.init.consumer_trig   = PIS_CH12_TIMER0_ITR0; 
	h_pis.init.consumer_clk    = PIS_CLK_PCLK1;
	ald_pis_create(&h_pis);
	
	h_pis.init.consumer_trig   = PIS_CH13_TIMER1_ITR1;
	ald_pis_create(&h_pis);

	pis_tim_start();

	while (1) {
		ald_delay_ms(1000);
	}
}
/**
  * @}
  */

/**
  * @}
  */
