/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
timer_handle_t h_tim0;

/* overflow times */
 int16_t encoder_overflow_count = 0;
/* motor direction */
__IO int8_t motor_direction = 0;
/* current count value */
 int32_t capture_count = 0;
/* last count value */
__IO int32_t last_count = 0;
/* shaft speed */
__IO float shaft_speed = 0.0f;

/* encoder resolution */
#define ENCODER_RESOLUTION              16
#define ENCODER_TOTAL_RESOLUTION        (ENCODER_RESOLUTION * 2)  

/* Timer overflow value */		
#define ENCODER_TIM_PERIOD              65535

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_init(void)
{
	gpio_init_t x;

	x.mode = GPIO_MODE_INPUT;
	x.pupd = GPIO_PUSH_DOWN;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_CMOS;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.func = GPIO_FUNC_2;
	
	ald_gpio_init(TIMER_ADT0_CH1_PORT, TIMER_ADT0_CH1_PIN, &x);
	ald_gpio_init(TIMER_ADT0_CH2_PORT, TIMER_ADT0_CH2_PIN, &x);
}

/**
  * @brief  ad16c4t0 time period elapsed function
  * @retval Status
  */
void ad16c4t0_period_elapsed_callback(struct timer_handle_s *arg)
{
	if (TIMER_GET_DIR(&h_tim0))
		encoder_overflow_count++;
	else
		encoder_overflow_count--;
}

/**
  * @brief  ad16c4t0 function
  * @retval Status
  */
void ad16c4t0_init(void)
{
	timer_encoder_init_t encoder_cfg;
	
	/* Initialize AD16C4T0 */
	h_tim0.perh           = AD16C4T0;
	h_tim0.init.prescaler = 0;
	h_tim0.init.mode      = TIMER_CNT_MODE_UP;
	h_tim0.init.period    = 0xFFFF;
	h_tim0.init.clk_div   = TIMER_CLOCK_DIV1;
	h_tim0.init.re_cnt    = 0;
        h_tim0.period_elapse_cbk = ad16c4t0_period_elapsed_callback;
		
	encoder_cfg.mode = TIMER_ENC_MODE_TI1;
	encoder_cfg.ic1_polarity = TIMER_IC_POLARITY_RISE;
	encoder_cfg.ic1_sel      = TIMER_IC_SEL_DIRECT;
	encoder_cfg.ic1_psc      = TIMER_IC_PSC_DIV1;
	encoder_cfg.ic1_filter   = 0;
	
	encoder_cfg.ic2_polarity = TIMER_IC_POLARITY_RISE;
	encoder_cfg.ic2_sel      = TIMER_IC_SEL_DIRECT;
	encoder_cfg.ic2_psc      = TIMER_IC_PSC_DIV1;
	encoder_cfg.ic2_filter   = 0;
	
	ald_timer_encoder_init(&h_tim0, &encoder_cfg);
	ald_mcu_irq_config(AD16C4T0_UP_IRQn, 0, 1, ENABLE);
	ald_timer_interrupt_config(&h_tim0, TIMER_IT_UPDATE, ENABLE);
}

/**
  * @brief  Main function
  * @retval Status
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
        ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1,48000000);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	
	timer_pin_init();
	ad16c4t0_init();
	ald_timer_encoder_start_by_it(&h_tim0, TIMER_CHANNEL_1);
	while(1) {
		ald_delay_ms(1000);
	}
}

/**
  * @brief  Systick callback function
  * @retval Status
  */
void ald_systick_irq_cbk(void)
{
	static uint16_t i = 0;
	
	++i;
	/* 100ms */
	if (i == 100) {
		/* Get counter direction */
		motor_direction = TIMER_GET_DIR(&h_tim0);
		/* total_count = counter_value + overflow_times * ENCODER_TIM_PERIOD  */
		capture_count = TIMER_GET_CNT(&h_tim0) + (encoder_overflow_count * ENCODER_TIM_PERIOD);
		/* shaft_speed = 100ms_counter_value / encoder_total_resolution * time_parameter  */
		shaft_speed = (float)(capture_count - last_count) / ENCODER_TOTAL_RESOLUTION * 10 ;
		 /* Current counter total value */
		last_count = capture_count;
		i = 0;
	}
}
/**
  * @}
  */
/**
  * @}
  */
