/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "main.h"
#include "string.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
  
uint32_t s_wdata[64] = {0};
uint32_t s_rdata[64] = {0};

qspi_dac_cfg_t g_dac;
qspi_handle_t hqspi;
qspi_data_capture_cfg_t g_capture_cfg = {0};


void qspi_pin_init(void);
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr);
ald_status_t bsp_w25q128_enable_quad_mode(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_disable_protect(qspi_handle_t *hqspi);


void config_flash_param(void)
{
	qspi_device_size_t l_dsize;
	
	l_dsize.cs0 = QSPI_NSS_512M;
	l_dsize.cs1 = QSPI_NSS_512M;
	l_dsize.cs2 = QSPI_NSS_512M;
	l_dsize.cs3 = QSPI_NSS_512M;
	l_dsize.addr = 2;   	/* 2+1 address bytes */
	l_dsize.page = 256;    
	l_dsize.blk = 16;   	
	
	ald_qspi_device_size_config(&hqspi, &l_dsize);    
}

/**
  * @brief Test main function
  * @param None
  * @retval None
  */
int main(void)
{
	int i;
	
	ald_cmu_init(); 
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);    
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);    
	ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HCLK2);   
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);   

	for (i = 0; i < 64; ++i) {
		s_wdata[i] = 0xF2;
		s_rdata[i] = 0x55;
	}

	qspi_pin_init();

	hqspi.perh          = QSPI;
	hqspi.init.clkdiv   = QSPI_DIV_8;
	hqspi.init.cpol     = QSPI_CPOL_H;    
	hqspi.init.chpa     = QSPI_CPHA_2E;
	hqspi.init.nssdcode = QSPI_SINGLE_CHIP;
	hqspi.init.chipsel  = QSPI_CS_NSS0;
	hqspi.init.wrppin   = DISABLE;
	
	config_flash_param();
	
	g_dac.dtrprtcol  = DISABLE;
	g_dac.ahbdecoder = DISABLE;
	g_dac.xipimmed   = DISABLE;
	g_dac.xipnextrd  = DISABLE;
	g_dac.addrremap  = DISABLE;
	g_dac.dmaperh    = DISABLE;
	
	g_dac.wrinit.instxfer = QSPI_XFER_SINGLE;   
	g_dac.wrinit.addxfer  = QSPI_XFER_SINGLE;
	g_dac.wrinit.datxfer  = QSPI_XFER_QUAD;
	g_dac.wrinit.autowel  = ENABLE;     		
	g_dac.wrinit.wrcde    = 0x32;  		
	g_dac.wrinit.dcyles   = 0;
	
	g_dac.rdinit.instxfer = QSPI_XFER_SINGLE;  
	g_dac.rdinit.addxfer  = QSPI_XFER_QUAD;
	g_dac.rdinit.datxfer  = QSPI_XFER_QUAD;
	g_dac.rdinit.ddrbit   = DISABLE;
	g_dac.rdinit.modebit  = ENABLE;  		
	g_dac.rdinit.mbitval  = 0xFF;
	g_dac.rdinit.rdcde    = 0xEB;  		
	g_dac.rdinit.dcyles   = 4;

	qspi_dac_config(&hqspi, &g_dac);
	
	/* if clkdiv == QSPI_DIV_2*/
#if 0
	if (hqspi.init.clkdiv == QSPI_DIV_2) {
		g_capture_cfg.smpledge = QSPI_FALLING_E;
		g_capture_cfg.dlydcl = 3;
		ald_qspi_read_data_capture_config(&hqspi, &g_capture_cfg);
	}
#endif
	
	if (bsp_w25q128_reset(&hqspi) != OK)
		while (1);
	
	if (bsp_w25q128_disable_protect(&hqspi) != OK)
		while (1);
	
	if (bsp_w25q128_enable_quad_mode(&hqspi) != OK)
		while (1);
	
	if (bsp_w25q128_erase_block(&hqspi, 0) != OK)
		while (1);
	
	for (i = 0; i < 64; ++i)
		ald_qspi_dac_wr(QSPI_MEMORY_ADDRESS + 4 * i, s_wdata[i]);
	
	while (1) {
		ald_delay_ms(500);
	}
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void qspi_pin_init(void)
{
	gpio_init_t x;
	
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_TTL;
	
	x.func  = GPIO_FUNC_6;
	x.mode  = GPIO_MODE_OUTPUT;
	ald_gpio_init(GPIOB, GPIO_PIN_12, &x); 
	ald_gpio_init(GPIOB, GPIO_PIN_13, &x); 
	
	x.func  = GPIO_FUNC_6;
	x.mode = GPIO_MODE_INPUT;
	ald_gpio_init(GPIOB, GPIO_PIN_14, &x); 
	ald_gpio_init(GPIOB, GPIO_PIN_15, &x); 
	ald_gpio_init(GPIOB, GPIO_PIN_10, &x); 
	ald_gpio_init(GPIOB, GPIO_PIN_11, &x); 
}

/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint8_t status;

	/* Initialize the reset enable command */
	s_command.code     = RESET_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;
	s_command.d_sycle  = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Send the reset memory command */
	s_command.code = RESET_MEMORY_CMD;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &status;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */
	if (!(status & 0x01))
		return OK;

	return ERROR;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint8_t status;

	/* Enable write operations */
	s_command.code     = WRITE_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;/* QSPI_MODEBIT_DISABLE; */
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		while (1);

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &status;

	do
	{
		if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
			return ERROR;
	} while (status & 0x01);   

	return OK;
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
	qspi_stig_cmd_t s_command = {0};
	uint8_t status;
	uint32_t tickstart = 0U;
	uint32_t timeout   = 1000U;

	/* Before Erase Sector,must send write enable */
	if (bsp_w25q128_wr_enable(hqspi) != OK)
		while (1);

	/* Enable write operations */
	s_command.code     = SECTOR_ERASE_CMD;
	s_command.addr_len = 0x3;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;	/* QSPI_MODEBIT_DISABLE; */
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;
	s_command.addr     = blkaddr;
	s_command.op_addr  = ENABLE;

	/* Send the command */
	while (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
	{
		if (timeout != ALD_MAX_DELAY) {
		    if ((timeout == 0) || ((ald_get_tick() - tickstart) > timeout))
			return ERROR;
		}
	}

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &status;
	s_command.wr_len   = 0;

	do
	{
		if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
			return ERROR;
	} while (status & 0x01);   

	return OK;
}

/**
  * @brief  Enable quad mode
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_enable_quad_mode(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint8_t status;

	status = 0x02;

	s_command.code   = WRITE_STATUS_REG2_CMD;
	s_command.wr_buf = &status;
	s_command.wr_len = 1;
	s_command.rd_len = 0;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code = READ_STATUS_REG1_CMD;
	s_command.rd_buf = &status;
	s_command.rd_len = 1;
	s_command.wr_len = 0;

	do
	{
		if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
			return ERROR;
	} while (status & 0x01);   

	s_command.code = READ_STATUS_REG2_CMD;
	s_command.rd_buf = &status;
	s_command.rd_len = 1;
	s_command.wr_len = 0;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	if ((status & 0x02) == 0)
		return ERROR;

	return OK;
}

/**
  * @brief  Disable protect 
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_disable_protect(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint8_t status;

	if (bsp_w25q128_wr_enable(hqspi) != OK)
		return ERROR;

	status = 0;
	s_command.code = WRITE_STATUS_REG1_CMD;
	s_command.wr_buf = &status;
	s_command.wr_len = 1;
	s_command.rd_len = 0;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &status;
	s_command.wr_len   = 0;

	do
	{
		if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
			return ERROR;
	} while (status & 0x01);   

	return OK;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
