/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
#define BUFFER_SIZE (32 * 1024 / 4)

qspi_handle_t hqspi;
uint32_t rdata[BUFFER_SIZE];
env_t env;

void qspi_pin_init(void);
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr);
void bsp_qspi_dac_init(qspi_handle_t *hqspi);
void uart_init(void);
void uart_recv_by_dma(uart_handle_t *hperh, uint32_t size, uint8_t channel);
void check_qspi_data(void);

/**
  * @brief  Transfer complete callback
  * @retval None
  */
void transfer_complete(void)
{
	env.flag = 1;
	return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_96M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);
	/* Enable all peripherals */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	/* Clear the env */
	memset(&env, 0x0, sizeof(env_t));
	/* Initialize the uart */
	uart_init();
	/* Initialize the QSPI pin */
	qspi_pin_init();
	/* Initialize the INDAC QSPI */
	bsp_qspi_dac_init(&hqspi);
	/* Erase 32Kbytes space */
	bsp_w25q128_erase_block(&hqspi, 0);

	/* Receive data from UART and store int in QSPI-FLASH by DMA */
	uart_recv_by_dma(&env.h_uart, BUFFER_SIZE * 4, 0);

	/* Waiting DMA transfer completed */
	while (env.flag == 0);
	/* Check the data */
	check_qspi_data();

	while (1) {
		ald_delay_ms(1000);
	}
}

/** @brief  Receive data from UART and store int in QSPI-FLASH by DMA.
  * @param  hperh: Pointer to the @ref uart_handle_t structure.
  * @param  size: Size of the data unit bytes.
  * @param  channel: DMA channel.
  * @return None
  */
void uart_recv_by_dma(uart_handle_t *hperh, uint32_t size, uint8_t channel)
{
	hperh->hdmarx.perh     = DMA0;
	hperh->hdmarx.cplt_cbk = NULL;
	hperh->hdmarx.cplt_arg = NULL;
	hperh->hdmarx.err_cbk  = NULL;
	hperh->hdmarx.err_arg  = NULL;

	env._len   = size > 1024 ? 1024 : size;
	env.len    = size;
	env.dma_ch = channel;
	env.cbk    = transfer_complete;

	ald_dma_config_struct(&hperh->hdmarx.config);
	hperh->hdmarx.config.src     = (void *)&hperh->perh->RXBUF;
	hperh->hdmarx.config.dst     = (void *)QSPI_MEMORY_ADDRESS;
	hperh->hdmarx.config.size    = env._len;
	hperh->hdmarx.config.src_inc = DMA_DATA_INC_NONE;
	hperh->hdmarx.config.dst_inc = DMA_DATA_INC_BYTE;
	hperh->hdmarx.config.msel    = DMA_MSEL_UART1;
	hperh->hdmarx.config.msigsel = DMA_MSIGSEL_UART_RNR;
	hperh->hdmarx.config.burst   = ENABLE;
	hperh->hdmarx.config.channel = channel;
	hperh->hdmarx.config.R_power = DMA_R_POWER_1;

	ald_dma_config_basic(&hperh->hdmarx);
	SET_BIT(hperh->perh->FCON, UART_FCON_RFRST_MSK);
	ald_uart_dma_req_config(hperh, UART_DMA_REQ_RX, ENABLE);

	return;
}

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_init(void)
{
	gpio_init_t x;

	/* Initialize tx pin */
	x.mode = GPIO_MODE_OUTPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_0, &x);

	/* Initialize rx pin */
	x.mode = GPIO_MODE_INPUT;
	x.odos = GPIO_PUSH_PULL;
	x.pupd = GPIO_PUSH_UP;
	x.podrv = GPIO_OUT_DRIVE_1;
	x.nodrv = GPIO_OUT_DRIVE_0_1;
	x.flt  = GPIO_FILTER_DISABLE;
	x.type = GPIO_TYPE_TTL;
	x.func = GPIO_FUNC_3;
	ald_gpio_init(GPIOC, GPIO_PIN_1, &x);

	memset(&env.h_uart, 0x0, sizeof(uart_handle_t));
	env.h_uart.perh             = UART1;
	env.h_uart.init.baud        = 115200;
	env.h_uart.init.word_length = UART_WORD_LENGTH_8B;
	env.h_uart.init.stop_bits   = UART_STOP_BITS_1;
	env.h_uart.init.parity      = UART_PARITY_NONE;
	env.h_uart.init.mode        = UART_MODE_UART;
	env.h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
	env.h_uart.tx_cplt_cbk      = NULL;
	env.h_uart.rx_cplt_cbk      = NULL;
	env.h_uart.error_cbk        = NULL;
	ald_uart_init(&env.h_uart);

	return;
}

/**
  * @brief  Write test data to W25Q128
  * @retval None
  */
void check_qspi_data(void)
{
	int32_t i;

	for (i = 0; i < BUFFER_SIZE; ++i)
		rdata[i] = ald_qspi_dac_rd(QSPI_MEMORY_ADDRESS + 4 * i);
}


/* BSP for W25Q128FV */
/**
  * @brief  Initializate pin of qspi module.
  * @retval None
  */
void qspi_pin_init(void)
{
	gpio_init_t x;

	/* Initialize scl pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_6;

	ald_gpio_init(QSPI_PORT, QSPI_NSS0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_SCK_PIN, &x);
	x.mode = GPIO_MODE_INPUT;
	ald_gpio_init(QSPI_PORT, QSPI_IO0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO1_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO2_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO3_PIN, &x);
}

/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf = 0xffffffff;

	/* Initialize the reset enable command */
	s_command.code     = RESET_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE; 
	s_command.d_sycle  = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Send the reset memory command */
	s_command.code = RESET_MEMORY_CMD;
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 3;
	s_command.addr     = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf & 0x01))
		return OK;

	return ERROR;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2] = {0xffffffff, 0xaaaaaaaa};

	/* Enable write operations */
	s_command.code     = WRITE_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;//QSPI_MODEBIT_DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		while(1);

	s_command.code	   = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf[0] & 0x01))
		return OK;

	return ERROR; 
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2]   = {0xffffffff, 0xaaaaaaaa};
	uint32_t tickstart = 0;
	uint32_t timeout   = 1000;

	/* Before Erase Sector,must send write enable */
	if (bsp_w25q128_wr_enable(hqspi) != OK)
		while(1);

	/* Enable write operations */
	s_command.code     = SECTOR_BLOCK0_CMD;
	s_command.addr_len = 0x3;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;//QSPI_MODEBIT_DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;
	s_command.addr     = blkaddr;
	s_command.op_addr  = ENABLE;
	/* Send the command */
	while (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK) {
		if (timeout != ALD_MAX_DELAY) {
			if ((timeout ==0) || ((ald_get_tick() - tickstart) > timeout)) 
				return ERROR;
		}
	}

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	ald_qspi_execute_stig_cmd(hqspi, &s_command);

	while (1) {
		if (rbuf[0] & 0x01)
			ald_qspi_execute_stig_cmd(hqspi, &s_command);
		else
			return OK;
	}
}

/**
  * @brief  Initializate indac
  * @param  hqspi: QSPI handle
  * @retval Status
  */
void bsp_qspi_dac_init(qspi_handle_t *hqspi)
{
	hqspi->perh         = QSPI;
	hqspi->init.clkdiv  = QSPI_DIV_4;
	hqspi->init.cpol    = QSPI_CPOL_L;
	hqspi->init.chpa    = QSPI_CPHA_1E;
	hqspi->init.chipsel = QSPI_CS_NSS0;
	hqspi->init.wrppin  = DISABLE;
	ald_qspi_init(hqspi);
	bsp_w25q128_reset(hqspi);
}
/**
  * @}
  */
/**
  * @}
  */
