/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
#define QSPI_ADDRESS (*(volatile uint32_t *)0x90000000)

#define BUFFER_SIZE 600
uint32_t wdata[BUFFER_SIZE] = {0};
uint32_t rdata[BUFFER_SIZE] = {0};
qspi_handle_t hqspi;
qspi_legacy_cfg_t stdspi;

void qspi_pin_init(void);
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr);
uint32_t tmp = 0;
qspi_auto_poll_t pollcfg;

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint16_t i = 0;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_PLL1);	/**< qspi clock 48Mhz */
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
	qspi_pin_init();

	for (i = 0; i < BUFFER_SIZE; ++i) {
		wdata[i] = BUFFER_SIZE-i;
		rdata[i] = 0x55555555;
	}

	/* step1: qspi basic config */
	hqspi.perh         = QSPI;
	hqspi.init.clkdiv  = QSPI_DIV_24;
	hqspi.init.cpol    = QSPI_CPOL_L;
	hqspi.init.chpa    = QSPI_CPHA_1E;
	hqspi.init.chipsel = QSPI_CS_NSS0;
	hqspi.init.wrppin  = DISABLE;
	ald_qspi_init(&hqspi);

	stdspi.tx_thrd	= 0x10;
	stdspi.rx_thrd	= 0x10;
	ald_qspi_legacy_config(&hqspi, &stdspi);

	bsp_w25q128_reset(&hqspi);
	bsp_w25q128_erase_block(&hqspi,0);
	
	// write enable
	(*(volatile uint8_t*)QSPI_MEMORY_ADDRESS) = 0x06;
	tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);

	//write data
	(*(volatile uint32_t*)QSPI_MEMORY_ADDRESS) = 0x00000002;
	(*(volatile uint32_t*)QSPI_MEMORY_ADDRESS) = 0xABCDEF00;
	tmp = (*(volatile uint32_t*)QSPI_MEMORY_ADDRESS);
	tmp = (*(volatile uint32_t*)QSPI_MEMORY_ADDRESS);

	while(READ_BIT(hqspi.perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES);
	ald_delay_ms(1000);

	//read data
	(*(volatile uint32_t*)QSPI_MEMORY_ADDRESS) = 0x03;
	(*(volatile uint32_t*)QSPI_MEMORY_ADDRESS) = 0x00000000;
	tmp = (*(volatile uint32_t*)QSPI_MEMORY_ADDRESS);
	tmp = (*(volatile uint32_t*)QSPI_MEMORY_ADDRESS);

	while (1) {
		ald_delay_ms(1000);
	}
}


/**
  * @brief  Initializate pin of qspi module.
  * @retval None
  */
void qspi_pin_init(void)
{
	gpio_init_t x;

	/* Initialize scl pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_CMOS;
	x.func  = GPIO_FUNC_6;

	ald_gpio_init(QSPI_PORT, QSPI_NSS0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_SCK_PIN, &x);
	x.mode = GPIO_MODE_INPUT;
	ald_gpio_init(QSPI_PORT, QSPI_IO0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO1_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO2_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO3_PIN, &x);
}

/* BSP for W25Q128FV */
/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
	
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
	uint32_t rbuf = 0xff;
	rbuf = rbuf;

	/* Initialize the reset enable command */
	(*(volatile uint8_t*)QSPI_MEMORY_ADDRESS) = RESET_ENABLE_CMD;
	while(READ_BIT(hqspi->perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES);

	/* Send the reset memory command */
	(*(volatile uint8_t*)QSPI_MEMORY_ADDRESS) = RESET_MEMORY_CMD;
	while(READ_BIT(hqspi->perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES);

	rbuf = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);
	rbuf = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);

	return OK;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
	uint8_t tmp = 0xff;

	/* Enable write operations */
	(*(volatile uint8_t*)QSPI_MEMORY_ADDRESS) = WRITE_ENABLE_CMD;
	 while(READ_BIT(hqspi->perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES);
	tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);

	(*(volatile uint16_t*)QSPI_MEMORY_ADDRESS) = 0x0005;
	tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);
	tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);

	if (tmp &0x02)
		return OK;

	return ERROR; 
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
	/* Before Erase Sector,must send write enable */
	if (bsp_w25q128_wr_enable(hqspi) != OK)
		while(1);

	/* Enable write operations */
	(*(volatile uint32_t*)QSPI_MEMORY_ADDRESS) = 0x00000020;
	 while(READ_BIT(hqspi->perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES);
	tmp = (*(volatile uint32_t*)QSPI_MEMORY_ADDRESS);

	/* Send the command */
	do {
		(*(volatile uint16_t*)QSPI_MEMORY_ADDRESS) = 0x0005;
		tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);
		tmp = (*(volatile uint8_t*)QSPI_MEMORY_ADDRESS);
	} while(tmp & 0x01);

	return OK;
}
/**
  * @}
  */
/**
  * @}
  */
