/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

qspi_handle_t hqspi;

void qspi_pin_init(void);
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr);
void bsp_qspi_indac_init(qspi_handle_t *hqspi);
ald_status_t bsp_qspi_indac_poll_transmit(qspi_handle_t *hqspi,uint32_t addr, uint8_t *srcbuf, uint32_t len);
ald_status_t bsp_qspi_indac_poll_receive(qspi_handle_t *hqspi,uint32_t saddr, uint8_t* srcbuf, uint32_t len);

#define BUFFER_SIZE 32

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint32_t i = 0;
	uint32_t wdata[BUFFER_SIZE];
	uint32_t rdata[BUFFER_SIZE];

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
	ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HRC);
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	for(i =0; i < BUFFER_SIZE; i++) {
		wdata[i] = 0x11223300 + i;
		rdata[i] = 0;
	}

	qspi_pin_init();
	bsp_qspi_indac_init(&hqspi);
	
	bsp_w25q128_reset(&hqspi);
	bsp_w25q128_erase_block(&hqspi, 0);
	
	if (bsp_qspi_indac_poll_transmit(&hqspi,0x00000000, (uint8_t *)wdata, 32) != OK) {
		while(1);
	}
	ald_delay_ms(1000);

	if (bsp_qspi_indac_poll_receive(&hqspi,0x00000000, (uint8_t*)rdata, 32) != OK) {
		while(1);
	}
	ald_delay_ms(1000);

	while (1) {
		ald_delay_ms(1000);
	}
}

/* BSP for W25Q128FV */
/**
  * @brief  Initializate pin of qspi module.
  * @retval None
  */
void qspi_pin_init(void)
{
	gpio_init_t x;

	/* Initialize scl pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_1;
	x.flt   = GPIO_FILTER_DISABLE;
	x.nodrv = GPIO_OUT_DRIVE_6;
	x.podrv = GPIO_OUT_DRIVE_6;
	x.type  = GPIO_TYPE_TTL;
	x.func  = GPIO_FUNC_6;

	ald_gpio_init(QSPI_PORT, QSPI_NSS0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_SCK_PIN, &x);
	x.mode = GPIO_MODE_INPUT;
	ald_gpio_init(QSPI_PORT, QSPI_IO0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO1_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO2_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO3_PIN, &x);
}

/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf = 0xffffffff;

	/* Initialize the reset enable command */
	s_command.code     = RESET_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE; 
	s_command.d_sycle  = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Send the reset memory command */
	s_command.code = RESET_MEMORY_CMD;
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 3;
	s_command.addr     = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf & 0x01))
		return OK;

	return ERROR;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2] = {0xffffffff, 0xaaaaaaaa};

	/* Enable write operations */
	s_command.code     = WRITE_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;//QSPI_MODEBIT_DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len	  = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		while(1);

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf[0] & 0x01))
		return OK;

	return ERROR;
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2]   = {0xffffffff, 0xaaaaaaaa};
	uint32_t tickstart = 0;
	uint32_t timeout   = 1000;

	/* Before Erase Sector,must send write enable */
	if (bsp_w25q128_wr_enable(hqspi) != OK)
		while(1);

	/* Enable write operations */
	s_command.code     = SECTOR_ERASE_CMD;
	s_command.addr_len = 0x3;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;
	s_command.addr     = 0x0;
	s_command.op_addr  = ENABLE;
	/* Send the command */
	while (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK) {
		if (timeout != ALD_MAX_DELAY) {
			if ((timeout ==0) || ((ald_get_tick() - tickstart) > timeout)) 
				return ERROR;
		}
	}

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	ald_qspi_execute_stig_cmd(hqspi, &s_command);

	while (1) {
		if (rbuf[0] & 0x01)
			ald_qspi_execute_stig_cmd(hqspi, &s_command);
		else
			return OK;
	}
}

/**
  * @brief  Initializate indac
  * @param  hqspi: QSPI handle
  * @retval Status
  */
void bsp_qspi_indac_init(qspi_handle_t *hqspi)
{
	qspi_write_cfg_t wrins = {0};
	qspi_read_cfg_t rdins = {0};

	/* step1: qspi basic config */
	hqspi->perh          = QSPI;
	hqspi->init.clkdiv   = QSPI_DIV_24;
	hqspi->init.cpol     = QSPI_CPOL_L;
	hqspi->init.chpa     = QSPI_CPHA_1E;
	hqspi->init.chipsel  = QSPI_CS_NSS0;
	hqspi->init.wrppin   = DISABLE;
	ald_qspi_init(hqspi);

	/* step2: qspi write instruction config */
	wrins.autowel  = ENABLE; 
	wrins.wrcde    = PAGE_PROG_CMD;
	wrins.dcyles   = 0x0;
	wrins.addxfer  = QSPI_XFER_SINGLE;
	wrins.datxfer  = QSPI_XFER_SINGLE;
	wrins.instxfer = QSPI_XFER_SINGLE;
	ald_qspi_write_config(hqspi, &wrins);

	/* step3: qspi read instruction config */
	rdins.rdcde    = READ_CMD;
	rdins.addxfer  = QSPI_XFER_SINGLE;
	rdins.datxfer  = QSPI_XFER_SINGLE;
	rdins.instxfer = QSPI_XFER_SINGLE;
	rdins.dcyles   = 0x0;
	rdins.ddrbit   = DISABLE;
	rdins.modebit  = DISABLE;
	ald_qspi_read_config(hqspi, &rdins);
}

/**
  * @brief  Transfer data
  * @param  hqspi: QSPI handle
  * @param  srcbuf: Pointer to data
  * @param  len: Length of data
  * @retval Status
  */
ald_status_t bsp_qspi_indac_poll_transmit(qspi_handle_t *hqspi,uint32_t addr, uint8_t *srcbuf, uint32_t len)
{
	qspi_indac_cfg_t idacfg = {0};

	/* qspi indirect init */
	idacfg.trgaddr = QSPI_MEMORY_ADDRESS;
	idacfg.srampt  = 0x7F;			/* Recommand: SRAM transmit partition bigger than the flash page size*/
	idacfg.trgrng  = 0x04;
	idacfg.rdwmark = 0x00000000;		/* Disable read watermark */
	idacfg.wrwmark = 0xffffffff;		/* Between 0 and flash page size */
	ald_qspi_indac_config(hqspi, &idacfg);

	return ald_qspi_indac_transmit_by_poll(hqspi, addr, srcbuf, len);	
}

/**
  * @brief  Receive data
  * @param  hqspi: QSPI handle
  * @param  srcbuf: Pointer to data
  * @param  len: Length of data
  * @retval Status
  */
ald_status_t bsp_qspi_indac_poll_receive(qspi_handle_t *hqspi,uint32_t saddr, uint8_t* srcbuf, uint32_t len)
{
	qspi_indac_cfg_t idacfg = {0};

	/* qspi indirect init */
	idacfg.trgaddr = QSPI_MEMORY_ADDRESS;
	idacfg.srampt  = 0x7F;
	idacfg.trgrng  = 0x04;
	idacfg.rdwmark = 0x00000000;
	idacfg.wrwmark = 0xffffffff;
	ald_qspi_indac_config(hqspi, &idacfg);

	return ald_qspi_indac_read_by_poll(hqspi,saddr, srcbuf, len);	
}
/**
  * @}
  */
/**
  * @}
  */
