/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */ 

#include <string.h>
#include "main.h"


/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
#define QSPI_ADDRESS (*(volatile uint32_t *)0x90000000)

uint32_t wdata[128] = {0};
uint32_t rdata[128] = {0};
qspi_handle_t hqspi;
qspi_dac_cfg_t g_dac;

void qspi_pin_init(void);
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi);
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr);

qspi_data_capture_cfg_t g_capture_cfg = {0};


/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
	uint32_t i = 0;

	/* Initialize ALD */
	ald_cmu_init();
	/* Configure system clock */
	ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
	ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
	ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HRC);	
	ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

	for (i = 0; i < 128; ++i) {
		wdata[i] = 0xA0A0A0A1 + i;
		rdata[i] = 0xBBCCDD55;
	}

	qspi_pin_init();
	
	hqspi.perh         = QSPI;
	hqspi.init.clkdiv  = QSPI_DIV_2;
	hqspi.init.cpol    = QSPI_CPOL_L;
	hqspi.init.chpa    = QSPI_CPHA_1E;
	hqspi.init.chipsel = QSPI_CS_NSS0;
	hqspi.init.wrppin  = DISABLE;
	ald_qspi_init(&hqspi);
	
	g_dac.dtrprtcol  = DISABLE;
	g_dac.ahbdecoder = DISABLE;
	g_dac.xipimmed 	 = DISABLE;
	g_dac.xipnextrd  = DISABLE;
	g_dac.addrremap  = DISABLE;
	g_dac.dmaperh 	 = DISABLE;
	
	g_dac.wrinit.instxfer = QSPI_XFER_SINGLE;  
	g_dac.wrinit.addxfer = QSPI_XFER_SINGLE;
	g_dac.wrinit.datxfer = QSPI_XFER_SINGLE;
	g_dac.wrinit.autowel = ENABLE;     	
	g_dac.wrinit.wrcde   = 0x2;  		
	g_dac.wrinit.dcyles  = 0;
	
	g_dac.rdinit.instxfer = QSPI_XFER_SINGLE;   
	g_dac.rdinit.addxfer  = QSPI_XFER_SINGLE;
	g_dac.rdinit.datxfer  = QSPI_XFER_SINGLE;
	g_dac.rdinit.ddrbit   = DISABLE;
	g_dac.rdinit.modebit  = DISABLE;  
	g_dac.rdinit.mbitval  = 0x00;
	g_dac.rdinit.rdcde    = 0x03;  
	g_dac.rdinit.dcyles   = 0;

	qspi_dac_config(&hqspi, &g_dac);  
	
	/* if QSPI_DIV_2, delay the read data capturing logic */
	g_capture_cfg.smpledge = QSPI_FALLING_E;
	g_capture_cfg.dlydcl = 3;
	ald_qspi_read_data_capture_config(&hqspi, &g_capture_cfg);

	bsp_w25q128_reset(&hqspi);
	bsp_w25q128_erase_block(&hqspi, 0);

	for (i = 0; i < 128; ++i)
		ald_qspi_dac_wr((QSPI_MEMORY_ADDRESS+4*i), wdata[i]);
		
	for (i = 0; i < 128; ++i)
		rdata[i] = ald_qspi_dac_rd(QSPI_MEMORY_ADDRESS+4*i);

	while (1) {
		ald_delay_ms(1000);
	}
}

/**
  * @brief  Initializate pin of qspi module.
  * @retval None
  */
void qspi_pin_init(void)
{
	gpio_init_t x;

	/* Initialize scl pin */
	x.mode  = GPIO_MODE_OUTPUT;
	x.odos  = GPIO_PUSH_PULL;
	x.pupd  = GPIO_PUSH_UP;
	x.nodrv = GPIO_OUT_DRIVE_20;
	x.podrv = GPIO_OUT_DRIVE_20;
	x.flt   = GPIO_FILTER_DISABLE;
	x.type  = GPIO_TYPE_CMOS;
	x.func  = GPIO_FUNC_6;

	ald_gpio_init(QSPI_PORT, QSPI_NSS0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_SCK_PIN, &x);
	x.mode = GPIO_MODE_INPUT;
	ald_gpio_init(QSPI_PORT, QSPI_IO0_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO1_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO2_PIN, &x);
	ald_gpio_init(QSPI_PORT, QSPI_IO3_PIN, &x);
}

/* BSP for W25Q128FV */
/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf = 0xffffffff;

	/* Initialize the reset enable command */
	s_command.code     = RESET_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE; 
	s_command.d_sycle  = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Send the reset memory command */
	s_command.code = RESET_MEMORY_CMD;
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;
	
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf & 0x01))
		return OK;

	return ERROR;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2] = {0xffffffff, 0xaaaaaaaa};

	/* Enable write operations */
	s_command.code     = WRITE_ENABLE_CMD;
	s_command.addr_len = 0;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;//QSPI_MODEBIT_DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;

	/* Send the command */
	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		while(1);

	s_command.code	   = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	if (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK)
		return ERROR;

	/* Configure automatic polling mode to wait the memory is ready */ 
	if (!(rbuf[0] & 0x01))
		return OK;

	return ERROR; 
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
	qspi_stig_cmd_t s_command = {0};
	uint32_t rbuf[2]   = {0xffffffff, 0xaaaaaaaa};
	uint32_t tickstart = 0;
	uint32_t timeout   = 1000;

	/* Before Erase Sector,must send write enable */
	if (bsp_w25q128_wr_enable(hqspi) != OK)
		while(1);

	/* Enable write operations */
	s_command.code     = SECTOR_ERASE_CMD;
	s_command.addr_len = 0x3;
	s_command.wr_len   = 0;
	s_command.mode_bit = DISABLE;//QSPI_MODEBIT_DISABLE;
	s_command.d_sycle  = 0;
	s_command.rd_len   = 0;
	s_command.addr     = 0x00000000;
	s_command.op_addr  = ENABLE;
	/* Send the command */
	while (ald_qspi_execute_stig_cmd(hqspi, &s_command) != OK) {
		if (timeout != ALD_MAX_DELAY) {
			if ((timeout ==0) || ((ald_get_tick() - tickstart) > timeout)) 
				return ERROR;
		}
	}

	s_command.code     = READ_STATUS_REG1_CMD;
	s_command.addr_len = 0x0;
	s_command.rd_len   = 0x01;
	s_command.rd_buf   = &rbuf;

	ald_qspi_execute_stig_cmd(hqspi, &s_command);

	while (1) {
		if (rbuf[0] & 0x01)
			ald_qspi_execute_stig_cmd(hqspi, &s_command);
		else
			return OK;
	}
}
/**
  * @}
  */
/**
  * @}
  */
